/* 	main.c: 
 *
 *	part of the martian_modem
 *
 *	description:
 *		parsing args, signals handling, monitoring
 *		pty/port/session loop
 *		named pipe /dev/martian for commands on the fly
 *
 *	Author: A. Chentsov
 * 	Copying: LGPL
 *
 */

#include <stdio.h>

#include <sys/mman.h>		/*** mlockall */
#include <string.h>		/*** memcmp */
#include <asm/types.h>
#include "../martian.h"

#include <signal.h>		/*** kbd interrupt */

#include <unistd.h>		// close
#include <stdlib.h>		// exit

#include <sys/types.h>

#include <ctype.h>		/*** isdigit */


#include <sched.h>		/*** sched fifo... */

#include <stdint.h>		/*** uintxx_t */

#include <pwd.h>		/*** option parse */
#include <grp.h>
#include <sys/stat.h>

#define _GNU_SOURCE
#include <getopt.h>
#include <sys/ioctl.h>	/*** cmds: kdce_rx */

#include "sysdep.h"

#include "main.h"

#define PREFIX	NULL
#include "log.h"

#include "mport.h"

#include "common.h"
#include "core.h"

#ifndef __stringify
#define __stringify_nomacro(x)	#x
#define __stringify(x)		__stringify_nomacro (x)
#endif

/***** global vars *******/
struct _config config;
struct martian_common *Mcb;
struct _port mport;
int worker_tid;

/*************************/

/***** some signatures *****/
void setup_kbd_interrupt (void); 
static void setup_daemon_signals (void); 
static void setup_segv (void); 

static int check_procfs (void) {
	return system ("grep -q '/proc ' /proc/mounts 2> /dev/null");
}

/*** setup for tsc ***/
void watch_setup (void )
{
	config.haveTSC = 0;

	if (check_procfs() != 0) {
		LOGWARN ("procfs on /proc is not found. No TSC.\n");
		return;
	}

	int status = system ("cat /proc/cpuinfo | grep flags | grep -q tsc 2> /dev/null");
	LOGDEBUG (3, "check tsc status %d\n", status);
	if (status == 0) {
		FILE *fp = popen ("grep '^cpu M' /proc/cpuinfo 2>/dev/null", "r");
		if (! fp)
			return ;

		int c;
		while ((c = fgetc (fp)) != EOF) 
			if (isdigit (c)) {
				ungetc (c, fp);
				break;
			}

		fscanf (fp, "%lf", & config.freq);
		LOGDEBUG (2, "TSC: %lf MHz\n", config.freq);
		pclose (fp);

		if (config.freq == 0) 
			return;

		config.haveTSC = 1;
	}
}

static const char *sched_policy_text[] = {
	"normal", "FIFO", "RR"
};

static int realtime_setup (void)
{
	if (mlockall(MCL_CURRENT) != 0) {
		LOGSYSERR (Debug, "mlockall");
		LOGWARN ("cannot lock pages\n");
	}

	struct sched_param p;
	p.sched_priority = sched_get_priority_min (SCHED_FIFO) + 1;

	/*  after this if we screw up we'd hog the cpu
	 *  either we should run interactively
	 *  or have shell with priority FIFO_MIN+1
	 */
	if (sched_setscheduler (0, SCHED_FIFO, &p) != 0) {
		LOGSYSERR (Debug, "sched_setscheduler");
		LOGWARN ("cannot raise priority\n");
		return -1;
	}
	else {
		sched_getparam (0, &p);
		LOGDEBUG (2, "main thread scheduled %s with prio %d\n", 
		sched_policy_text[sched_getscheduler (0)], 
		p.sched_priority);
	}
	return 0;

}

void usage (char *argv[]) {
	fprintf (stdout, "Usage: %s [Options] [TTY]\n", argv[0]);
	fprintf (stdout, "Options:\n");
	fprintf (stdout, "   --daemon           \tRun as daemon\n");
	fprintf (stdout, "   --log=FILE         \tLog to FILE\n");
	fprintf (stdout, "   --syslog           \tUse syslog for logging\n");
	fprintf (stdout, "   --realtime         \tRaise priority of threads to realtime\n");
	fprintf (stdout, "   --smp              \tTrue smp mode\n");
	fprintf (stdout, "   --user=<name>      \tGive tty ownership to user <name>\n");
	fprintf (stdout, "   --group=<name>     \tChange tty group to <name>\n");
	fprintf (stdout, "   --country=<country>\n");
	fprintf (stdout, "   --no-cdclose       \tKeep working with client when carrier lost.\n");
	fprintf (stdout, "   --hide-pty         \tSave pty from others as soon its open. Client should notify he's here writing to device. For callback feature.\n");
	fprintf (stdout, "   --debug=LEVEL      \tEnable debugging. LEVEL = 1..5\n");
	fprintf (stdout, "   --info countries   \tPrint support info\n");
	fprintf (stdout, "   --help, --usage    \tPrint this message \n");
	fprintf (stdout, "TTY is the file name to access the modem, /dev/ttySM0 by default\n");

	/* for good */
	exit (1);
}

static void fix_id_to_code (void ) 
{
	extern __u16 OldCountryIdToCountryCode[0x56];
	struct {
		__u16 code;
		__u8 id;
	} IdtoCodeFix[] = {
		{ 0xfe, 0x15 }, /* Taiwan */
		{ 0xc6, 0x26 }, /* Slovenia */
		{ 0xc5, 0x27 }, /* Slovakia */
		{ 0xd0, 0x2a }, /* Europe */
		{ 0xfa, 0x3e }, /* Virgin Islands, U.S. */
		{ 0x1e, 0x4e }, /* Belarus */
		{ 0xc7, 0x4f }, /* Croatia */
		{ 0xf7, 0x50 }, /* Lithuania */
		{ 0xf8, 0x51 }, /* Estonia */
		{ 0xf9, 0x52 }  /* Latvia */
	};
	int idx;

	for (idx = 0; idx < sizeof IdtoCodeFix / sizeof IdtoCodeFix[0]; idx++) 
		OldCountryIdToCountryCode[IdtoCodeFix[idx].id] = IdtoCodeFix[idx].code;

}

/*
{ "Slovenia", "si", 0xc6, 0x26 }
{ "Slovakia", "sk", 0xc5, 0x27 }
{ "Europe", 0xd0, 0x2a }
{ "VIRGIN ISLANDS, U.S.", "vi", 0xfa, 0x3e }
{ "Belarus", 0x1e, 0x4e }
{ "Croatia", "hr", 0x4f }
{ "Lithuania", "lt", 0xf7, 0x50 }
{ "Estonia", "ee", 0xf8, 0x51 }
{ "Latvia", "lv", 0xf9, 0x52 }
{ "Taiwan", "tw", 0xfe, 0x15 }
*/

const struct {
	char *country;
	char element[3];
	unsigned char code;
} countries[] = {
	{ "Japan", "jp", 0 },
	{ "Argentina", "ar", 7 },
	{ "Australia", "au", 9 },
	{ "Austria", "at", 0x0a },
	{ "Bangladesh", "bd", 0x0d },
	{ "Barbados", "bb", 0x0e },
	  { "Belarus", "by", 0x1e },
	{ "Bolivia", "bo", 0x14 },
	{ "Brazil", "br", 0x16 },
	{ "Bulgaria", "bg", 0x1b },
	{ "Canada", "ca", 0x20 },
	{ "Chile", "cl", 0x25 },
	{ "China", "cn", 0x26 },
	{ "Colombia", "co", 0x27 },
	{ "Costa Rica", "cr", 0x2b },
	  { "Croatia", "hr", 0x4f },
	{ "Cyprus", "cy", 0x2d },
	{ "Czech Republic", "cz", 0x2e },
	{ "Denmark", "dk", 0x31 },
	{ "Ecuador", "ec", 0x35 },
	{ "Egypt", "eg", 0x36 },
	  { "Estonia", "ee", 0xf8 },
	  { "Europe", "eu", 0xd0 }, /* pseudo */
	{ "Finland", "fi", 0x3c },
	{ "France", "fr", 0x3d },
	{ "Germany", "de", 0x42 },
	{ "Greece", "gr", 0x46 },
	{ "Guam", "gu", 0x48 },
	{ "Guatemala", "gt", 0x49 },
	{ "Hong Kong", "hk", 0x50 },
	{ "Hungary", "hu", 0x51 },
	{ "Iceland", "is", 0x52 },
	{ "India", "in", 0x53 },
	{ "Indonesia", "id", 0x54 },
	{ "Ireland", "ie", 0x57 },
	{ "Israel", "il", 0x58 },
	{ "Italy", "it", 0x59 },
	{ "Korea, Republic of", "kr", 0x61 },
	{ "Kuwait", "kw", 0x62 },
	{ "Latvia", "lv", 0xf8 },
	{ "Lebanon", "lb", 0x64 },
	{ "Liechtenstein", "li", 0x68 },
	  { "Lithuania", "lt", 0xf7 },
	{ "Luxembourg", "lu", 0x69 },
	{ "Malaysia", "my", 0x6c },
	{ "Malta", "mt", 0x70 },
	{ "Mexico", "mx", 0x73 },
	{ "Morocco", "ma", 0x77 },
	{ "Netherlands", "nl", 0x7b },
	{ "New Zealand", "nz", 0x7e },
	{ "Nicaragua", "ni", 0x7f },
	{ "Norway", "no", 0x82 },
	{ "Oman", "om", 0x83 },
	{ "Pakistan", "pk", 0x84 },
	{ "Panama", "pa", 0x85 },
	{ "Paraguay", "py", 0x87 },
	{ "Peru", "pe", 0x88 },
	{ "Philippines", "ph", 0x89 },
	{ "Poland", "pl", 0x8a },
	{ "Portugal", "pt", 0x8b },
	{ "Puerto Rico", "pr", 0x8c },
	{ "Romania", "ro", 0x8e },
	{ "Russia", "ru", 0xb8 },
	{ "Saudi Arabia", "sa", 0x98 },
	{ "Singapore", "sg", 0x9c },
	{ "Slovakia", "sk", 0xc5 },
	{ "Slovenia", "si", 0xc6 },
	{ "South Africa", "za", 0x9f },
	{ "Spain", "es", 0xa0 },
	{ "Sri Lanka", "lk", 0xa1 },
	{ "Sweden", "se", 0xa5 },
	{ "Switzerland", "ch", 0xa6 },
	  { "Taiwan", "tw", 0xfe },
	{ "Thailand", "th", 0xa9 },
	{ "Tunisia", "tn", 0xad },
	{ "Turkey", "tr", 0xae },
	{ "Ukraine", "ua", 0xb2 },
	{ "United Arab Emirates", "ae", 0xb3 },
	{ "United Kingdom", "uk", 0xb4 },
	{ "United States", "us", 0xb5 },
	{ "Uruguay", "uy", 0xb7 },
	{ "Venezuela", "ve", 0xbb },
	{ "Viet Nam", "vn", 0xbc },
	  { "Virgin Islands, U.S.", "vi", 0xfa },
	{ .country = NULL }
};

static int mcountry_match (char *test, const char *country) 
{
	char *p = strchr (country, ',');
	if (p) {
		int idx = p - country;
		if (strncasecmp (country, test, idx) != 0)
			return 0;

		return (test[idx] == '\0') || (strcasecmp (country + idx, test + idx) == 0);
	}
	else 
		return strcasecmp (test, country) == 0;
}

#define OptItemMax	17

static int optslot (int opt) {
	int shift = 0;
	if (opt > 7) {
		shift++;
	}
	return opt - 2 + shift;
}

int parse_arguments (int argc, char *argv[])
{
	static struct option options[] = {
		{ "debuglevel", required_argument, NULL, 2 },
		{ "realtime", no_argument, NULL, 3 },
		{ "syslog", no_argument, NULL, 4 },
		{ "daemon", no_argument, NULL, 5 },
		{ "log", required_argument, NULL, 6 },
		{ "help", no_argument, NULL, 7 },
		{ "usage", no_argument, NULL, 7 },
		{ "no-cdclose", no_argument, NULL, 9 },
		{ "hide-pty", no_argument, NULL, 10 },
		{ "smp", no_argument, NULL, 11 },
 		{ "user", required_argument, NULL, 13 },
 		{ "group", required_argument, NULL, 14 },
 		{ "mode", required_argument, NULL, 15 },
 		{ "country", required_argument, NULL, 16 },
		{ "info", required_argument, NULL, 17 },
		/* when adding new - update OptItemMax above */
		{ NULL, }
	};

	config.debuglevel 	= 0;
	config.realtime		= 0;
	config.syslog		= 0;
	config.daemon		= 0;
	config.logfile 		= 0;
	config.check_carrier	= 1;
	config.symlink_name 	= "/dev/ttySM0";
	config.hide_pty		= 0;
	config.true_smp		= 0;
	
	config.permissions	= 0;
	config.mode		= 0;

	config.uid		= -1;
	config.gid		= -1;

	fix_id_to_code(); 

	opterr = 0;
	do {
		int optindex;
		int choice = getopt_long (argc, argv, "", options, &optindex);

		switch (choice) {
		case 3:
			config.realtime = 1;
			break;


		case 6:
			if (config.syslog) {
				LOGERR ("--log `file' conflicts with --syslog\n");
				return 0;
			}
			config.logfile = 1;
			switch (log_redirect_to_file (optarg)) {
			case -1:
				log_setup (Syslog);
				LOGWARN ("Failed to redirect to file. Using syslog\n");
				break;
			case -2:
				/**/
				LOGWARN ("Bad filename %s. Log is not redirected.\n", optarg);
				break;
			case 0:
				/* OK */
				;
			}
			config.logfilename = optarg;
			break;

		case 2:
			config.debuglevel = atoi (optarg);
			if (config.debuglevel < 0) {
				LOGWARN ("wrong log level %d, ignored\n", config.debuglevel);
				config.debuglevel = 0;
			}
			break;

		case 5:
			config.daemon = 1;
			/* implies syslog unless --log file*/

			break;

		case 4:
			if (config.logfile) {
				LOGERR ("--syslog conflicts with --log `file'\n");
				return 0;
			}
			config.syslog = 1;
			break;

		case 7:
			/* may be usage as well */
			usage (argv);
			return 1;

		case 9:
			config.check_carrier = 0;
			break;

		case 10:
			config.hide_pty = 1;
			break;

		case 11:
			config.true_smp = 1;
			break;


		case 13:
		{
			struct passwd *user = getpwnam (optarg);
			
			if (! user) {
				LOGWARN ("bad user name '%s', ignored\n", optarg);
			} else 
				config.uid = user->pw_uid;

			break;
		}

		case 14:
		{
			struct group *group = getgrnam (optarg);
			
			if (! group) 
				LOGWARN ("bad group '%s', ignored\n", optarg);
			else 
				config.gid = group->gr_gid;

			break;
		}

		case 15:
			config.mode = strtol (optarg, NULL, 0);
			if (config.mode & ~(S_IRWXU | S_IRWXG | S_IRWXO)) {
				LOGWARN ("wrong mode for tty 0%o\n", config.mode);
				config.permissions = 0;
			}
			else {
				config.permissions = 1;
			}
			break;
		
		case 16:
		{
			int i = 0;
			while (countries[i].country) {
				if (mcountry_match (optarg, countries[i].country)
				   	    || strcasecmp (optarg, countries[i].element) == 0) {
					extern __u16 OldCountryIdToCountryCode[0x56];
					int id = 0;
					/*config.country_code = countries[i].code;*/
					for (; id < 0x56; id++) {
						if (OldCountryIdToCountryCode[id] == countries[i].code) {
							LOGDEBUG (Note, "country %s, set code %x, id %x\n", countries[i].country, countries[i].code, id);
							config.country_id = id;
							break;
						}
					}
					if (id == 0x56) {
						LOGWARN ("unsupported country %s\n", optarg);
						config.country_id = 0x19;
					}
					break;
				}
				i++;
			}
			
			if (! countries[i].country) {
				LOGWARN ("unsupported country %s, code not found\n", optarg);
				config.country_id = 0x19;
			}
			break;
		}

		case 17:
			LOGDEBUG (Note, "optarg: %s\n", optarg ? optarg : "NULL");
			if (optarg && (strcmp (optarg, "countries") == 0)) {
				int idx;
				printf ("Supported countries:\n");
				for (idx = 0; countries[idx].country; idx++) 
					printf ("%s\t%s\n", countries[idx].element, countries[idx].country);
			}
			else 
				LOGWARN ("unrecognized %s\n", optarg);

			/* for good */
			exit (1);

		case -1:
			if (optind + 1 == argc) 
				config.symlink_name = argv[optind];
			
			else if (optind != argc)
				LOGWARN ("trailing arguments ignored\n");

			return 1;

		/* option error */
		case ':':
			if (optind < 1 || argc < optind) 
				;
			else if (optopt != 0) {
				if (optopt <= OptItemMax && options[optslot (optopt)].has_arg == required_argument) 
					fprintf (stderr, "%s: needed parameter for option --%s\n", argv[0], options[optslot(optopt)].name);
			}
			return 0;

		case '?':
			/* may be usage as well */

			LOGDEBUG (Note, "optind: %d, optopt: '%c'=%d\n", optind, optopt, optopt);
			if (optind < 1 || argc < optind) 
				;
			else if (optopt != 0) {
				if (optopt <= OptItemMax && options[optslot (optopt)].has_arg == required_argument) 
					fprintf (stderr, "%s: needed parameter for option --%s\n", argv[0], options[optslot (optopt)].name);
				else 
					fprintf (stderr, "%s: unknown option -%c\n", argv[0], optopt);
			}
			else if (!strncmp (argv[optind-1], "--", 2)) {
				fprintf (stderr, "%s: unknown option %s\n", argv[0], argv[optind-1]);
			}
			else 
				fprintf (stderr, "%s: unexpect arg %s\n", argv[0], argv[optind - 1]);
			fprintf (stderr, "Do `%s --help' for usage\n", argv[0]);
			return 0;
		// default:
		}

        } while (1);

}

static void *serve_port (void *arg); 

#ifndef sigev_notify_thread_id
# define sigev_notify_thread_id 	_sigev_un._tid
#endif

#define SIG_PTY_TIMER 	SIGRTMIN
mtimer_t ptytimer;
/* pin_setup_timer
 *	set up for periodic task on pty
 *	called the do_io starts
 */
static int setup_pin_timer () {
	struct sigevent ptyevent;

	/* set up event */

	memset (&ptyevent, 0, sizeof ptyevent);

	// ptyevent.sigev_value.sival_int = pty; 
	ptyevent.sigev_signo = SIG_PTY_TIMER;

	/* works with syscalls but fails with glibc implementation */
	ptyevent.sigev_notify = SIGEV_THREAD_ID;
	ptyevent.sigev_notify_thread_id = worker_tid; /* gettid(); */

	int res = mtimer_create (CLOCK_REALTIME, &ptyevent, &ptytimer);
	if (res >= 0) {
		LOGDEBUG (2, "set timer for thread\n");
		return 0;
	}


	LOGWARN ("setting timer 0x%x by thread %d failed\n", ptytimer.id, mgettid());
	LOGSYSERR (Debug, "timer");
	
	/*** second try ***/
	memset (&ptyevent, 0, sizeof ptyevent);

	// ptyevent.sigev_value.sival_int = pty; 
	ptyevent.sigev_signo = SIG_PTY_TIMER;
	ptyevent.sigev_notify = SIGEV_SIGNAL;
	/***/
	res = mtimer_create (CLOCK_REALTIME, &ptyevent, &ptytimer);

	if (res < 0) {
		LOGERR ("cannot setup timer 0x%x, hangup won't work\n", ptytimer.id);
		return res;
	}

	LOGDEBUG (2, "set timer signal\n");
	return 0;
}

struct _state {
	uint8_t  dp_dsp_status;
	uint16_t io_uart_msr;
	uint16_t io_uart_status;
	unsigned x_modem_state;
	unsigned x_modem_mode;
	unsigned io_app_tx_count;

	unsigned x_line_rate;
	/* fifo sizes */
	int io_dte_tx_bytes;
	int io_dte_rx_bytes, io_dce_tx_bytes, io_dce_rx_bytes, io_app_tx_bytes;
} corestate;

extern unsigned x_modem_state;
extern unsigned x_modem_mode;
extern unsigned x_line_rate;
extern uint8_t  dp_dsp_status;
extern uint16_t dp_wDspRetrainState;

//extern char *sRate[];

unsigned first (unsigned ov, unsigned nv) { return ov; }
unsigned second (unsigned ov, unsigned nv) { return nv; }

char *rate1 (unsigned or, unsigned nr) { return sRate[or]; }
char *rate2 (unsigned or, unsigned nr) { return sRate[nr]; }

unsigned bits_set (unsigned oval, unsigned nval) { return ~oval & nval; }
unsigned bits_cleared (unsigned oval, unsigned nval) { return (oval & ~nval); }

#define CHECK_TRANSITION(var, fmt, diff1, diff2) \
	if (var != corestate.var) { 		\
		if (c) { ptr += sprintf (ptr, ", "); } else {ptr += sprintf (buf, "monitor: "); c = 1; }	\
		ptr += sprintf (ptr, __STRING (var) ": " fmt, diff1 (corestate.var, var), diff2 (corestate.var, var)); \
		corestate.var = var; 		\
	} \

#define CFIFO_DECLARE(fifo,type) 				\
extern type *fifo##_wptr, *fifo##_rptr, *fifo##_eptr, *fifo##_sptr

#define CFIFO_SIZE(fifo) 	(fifo##_eptr - fifo##_sptr)
#define CFIFO_BYTES(fifo) 				\
	(fifo##_wptr >= fifo##_rptr) ? 			\
		(fifo##_wptr - fifo##_rptr) :		\
		(fifo##_eptr - fifo##_rptr) + 		\
			(fifo##_wptr - fifo##_sptr)

#define CFIFO_CHANGED(fifo) (fifo##_bytes != corestate.fifo##_bytes)

#define CHECK_CFIFO_TRANSITION(fifo) 		\
{						\
	fifo##_bytes = CFIFO_BYTES (fifo);	\
	if (CFIFO_CHANGED(fifo)) {		\
		if (c) { ptr += sprintf (ptr, ", "); } else {ptr += sprintf (buf, "monitor: "); c = 1; }	\
		ptr += sprintf (ptr, __STRING (fifo) " %d/%d", fifo##_bytes, CFIFO_SIZE (fifo));	\
		corestate.fifo##_bytes = fifo##_bytes;	\
	}					\
}

CFIFO_DECLARE (io_dte_tx, unsigned char);
CFIFO_DECLARE (io_dte_rx, unsigned char);
CFIFO_DECLARE (io_app_tx, unsigned char);
CFIFO_DECLARE (io_dce_rx, unsigned short);
CFIFO_DECLARE (io_dce_tx, unsigned short);
static int io_dte_tx_bytes, io_dte_rx_bytes, io_dce_tx_bytes, io_dce_rx_bytes, io_app_tx_bytes;

void monitor_state (void ) {
	int c = 0;
	char buf[1024], *ptr = buf;
	CHECK_TRANSITION (io_uart_msr, "+%x -%x", bits_set, bits_cleared);
	CHECK_TRANSITION (io_uart_status, "+%x -%x", bits_set, bits_cleared);
	CHECK_TRANSITION (x_modem_state, "%d->%d", first, second);
	CHECK_TRANSITION (x_modem_mode, "%d->%d", first, second);
	CHECK_TRANSITION (x_line_rate, "%s->%s", rate1, rate2);
	CHECK_TRANSITION (dp_dsp_status, "+%x -%x", bits_set, bits_cleared);
	CHECK_TRANSITION (io_app_tx_count, "%d->%d", first, second);
	if (c == 1) {
		ptr += sprintf (ptr, "\n");
		LOGDEBUG (Event,  buf);
	}
	/* fifos */
	ptr = buf; c = 0;
	CHECK_CFIFO_TRANSITION (io_app_tx);
	CHECK_CFIFO_TRANSITION (io_dte_tx);
	CHECK_CFIFO_TRANSITION (io_dte_rx);
	CHECK_CFIFO_TRANSITION (io_dce_tx);
	CHECK_CFIFO_TRANSITION (io_dce_rx);
	if (c == 1) {
		ptr += sprintf (ptr, "\n");
		LOGDEBUG (Event,  buf);
	}
}

extern unsigned pin_check_mctrl (int changes);
static void pty_pulse (int signum) {
	static unsigned count = 0;

	logdebugadd (5, "pty<%d>\n", mgettid());
	mport_mctrl_flipflop (&mport, pin_check_mctrl (1));

	if (Debugged (Event))
		monitor_state();
	if (++count == 2) {
		//monitor_state();
		count = 0;
	}
}

void setup_pin_watcher (void) {
	struct sigaction timer_action;

	timer_action.sa_handler = pty_pulse;
	timer_action.sa_flags = 0;
	sigemptyset (&timer_action.sa_mask);
	sigaction (SIG_PTY_TIMER, &timer_action, NULL);
}

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
int get_cmd (FILE *control, char *cmd, int size) 
{
	char *ptr = cmd;
	int skipping = 0;
	int left = size - 1;
	int c;

	if (left < 0)
		return left;

	while (1) {
		errno = 0;
		c = getc (control);

		switch (c) {
		case ' ':
		case '\t':
			continue;

		case EOF:
			logdebugadd (Note, "EOF");
			if (errno == EINTR) {
				logdebugadd (2, "wa");
				continue;
			}
			if (feof (control))
				return 0;

			LOGSYSERR (Warning, "unexpected gets");
			continue;

		case '\n':
			if (skipping) {
				ptr = cmd;
				left = size - 1;
				skipping = 0;
				continue;
			}
			
			if (ptr == cmd) 
				continue;

			*ptr = '\0';
			return ptr - cmd;

		default:
			if (skipping)
				continue;

			if (left == 0) {
				LOGWARN ("too long cmd, skipping\n");
				skipping = 1;
			}
			else {
				*ptr++ = c;
				left--;
			}
		}
	}
}

int main (int argc, char *argv[]) 
{
	log_setup (Std); /* to std */

	if (! parse_arguments (argc, argv)) 
		return 1;
	
	if (config.syslog)
		log_setup (Syslog);

	if (config.daemon) {
		if (daemon (1, config.logfile) == -1) {
			/* log_setup (Std); */ /* default */
			LOGSYSERR (Debug, "daemon");
			LOGWARN ("failed to go daemon\n");
			config.daemon = 0;
		}
		else {
			if (config.daemon && ! config.logfile) {
				config.syslog = 1;
				log_setup (Syslog);
			}
			LOGDEBUG (2, "went daemon\n");
			/* set up signal */
			setup_daemon_signals(); 
		}
	}

#ifndef MSTAMP
# warning "MSTAMP undefined. Badly released."
#endif

	LOGDEBUG (Note, "source stamp `%s'\n", __stringify (MSTAMP));
	if (argc > 0)
	{
		int ai, quote;
		char cmd[256], *ptr;
		int size;

		size = sizeof cmd; ptr = cmd; quote = 0;

		SPRINTF_ADJUST (ptr, size, "%s", argv[1]);
		for (ai = 2; size && (ai < argc); ai++) {
			if (quote) {
				SPRINTF_ADJUST (ptr, size, " `%s'", argv[ai]);
				quote = 0;
			}
			else
				SPRINTF_ADJUST (ptr, size, " %s", argv[ai]);

			if (strncmp (argv[ai], "--country", strlen (argv[ai])) == 0 ||
			    strncmp (argv[ai], "--log", strlen (argv[ai])) == 0)
				quote = 1;
		}

		LOGDEBUG (Note, "arguments \"%s\"\n", cmd);
	}
	
	LOGDEBUG (Note, "configuration: debuglevel=%d%s\n", 
				config.debuglevel,
				config.realtime ? ", realtime" : ""
		);

	watch_setup();

	int res = mport_init (&mport);
	if (res != PortOk) 
		return res;
	
	/* for clean-ups */
	setup_kbd_interrupt();
	setup_segv();

	worker_tid = mgettid();
	/* setup in worker context */
	setup_pin_timer(); 
	setup_pin_watcher(); 
	
	/* launch communication thread */
	if (mthread_create (serve_port, NULL) == -1) {
		LOGERR ("Cannot create servicing thread.\n");
		return 4;
	}
	LOGDEBUG (Note, "servicing thread launched\n");

	// process_timer(); 		// smp case
	
#ifdef MARTIAN_CORE_DEBUG
	ask_driver_vars (dev_fd); 
#endif

	/* ready to serve */
	LOGINFO ("Your port is %s\n", config.symlink_name);

	/* worker */
	LOGDEBUG (2, "worker thread going into loop (tid = %d)\n", worker_tid);
	/* test spurious */
	/* sleep (3); */
	/* lt_add_timer ();*/
#define	MARTIAN_CONTROL	"/dev/martian"
	if (mkfifo (MARTIAN_CONTROL, 0600) == -1) {
		if (errno == EEXIST)
			LOGDEBUG (Note, MARTIAN_CONTROL " already exists\n");
		else 
			LOGSYSERR (Warning, "mkfifo");
	}

	FILE *control;

	do {
		control = fopen (MARTIAN_CONTROL, "r");
		if (control || errno != EINTR) 
			break;
		logdebugadd (Mechanism,"ci");
	} while (1);

	if (! control) {
		LOGSYSERR (Warning, "open");
		LOGWARN ("Cannot setup controlling node. No commands to be processed\n");
	}

  	while (1) {
		char cmd[256];
  		
		if (control) {
			int num = get_cmd (control, cmd, sizeof cmd);
			logdebugadd (Note, "out");
			if (num == 0) {
				LOGDEBUG (Note, "Reopenning control stream\n");
				fclose (control);
				do {
					control = fopen (MARTIAN_CONTROL, "r");
					if (control || errno != EINTR) 
						break;
					logdebugadd (Mechanism,"ci");
				} while (1);

				if (! control) {
					LOGSYSERR (Warning, "open");
					LOGWARN ("Cannot setup controlling node. No commands processed\n");
				}
				continue;
			}
			if (strcmp ("debug:dce_rx", cmd) == 0) {
				extern void iopl (int );
				LOGINFO ("misusing io_dce_rx_rptr\n");
				/*void *(*ptr) = &io_dce_rx_rptr;*/
				char *ptr = (char *) &io_dce_rx_rptr;
				iopl (3);asm volatile ("cli");
				
				//*(ptr + 2) = '\0';
				*((__u32 *)(ptr+1)) = 0;
				*((__u16 *)(ptr+3)) = '\0';

				asm volatile ("movl %esi,%esi");
				asm volatile ("sti"); iopl (0);
			}
			else if (strcmp ("debug:kdce_rx", cmd) == 0) {
				extern int mfd;
				LOGINFO ("misusing io_dce_rx_rptr through kernel\n");
				int result = ioctl (mfd, MARTIAN_SCREWUP_DCE_RX);
				if (result != 0)
					LOGSYSERR (Warning, "ioctl");
			}
			else if (strcmp ("debug:ff", cmd) == 0) {
				extern int mfd;
				LOGINFO ("initiating kmodule fifo fault\n");
				int result = ioctl (mfd, MARTIAN_FIFO_FAULT);
				if (result != 0)
					LOGSYSERR (Warning, "ioctl");
			}
			else {
				LOGDEBUG (Note, "unknown cmd <%s>\n", cmd);
			}
		}

		else
			pause();
	
		logdebugadd (5, "wa");
		
	} 
	return 0;
}

extern void pin_reset (int pin);
extern int pin_setup (int pin);
static void *serve_port (void *arg) 
{
	int server = -1;

		// setup_pin_watcher(); 

	LOGDEBUG (2, "servicing thread started (tid = %d)\n", mgettid());

	if (config.realtime)
		realtime_setup();

	mport_tinit (&mport);

	while (1) {
		int client;

		server = pin_setup (server);
		if (server == -1) {
			LOGERR ("cannot setup streams on user side\n");
			exit (1);
		}

		client = accept_client (server);
		if (client < 0) {
			LOGERR ("accept");
			continue; //??
		}
		LOGDEBUG (Sequence, "got a client\n");

		session_run (client);

		pin_close (client);
		LOGDEBUG (Sequence, "disconnected\n");
#ifdef MARTIAN_CORE_DEBUG
		print_core_vars_common (common);
		ask_driver_vars (mfd); 		// let's kernel driver dump them too
#endif
	}
	/* Never happen */
	pin_reset (server);
}

static void dump_global () {
	mport_dump_summary (&mport);
	/* */
}

static void mexit (void ) {
	extern void dump_isr (void);
	extern void show_ioctl_profile(void);
	extern void dump_dspop_profile(void );
	/* clean up */
	pin_reset (-1);

	if (Debugged (Stats)) {
		dump_global();
		dump_isr();
		show_ioctl_profile();
	}
	
	if (config.daemon) 
		LOGINFO ("Daemon stopped.\n");
	else 
		LOGINFO ("Bye.\n");
	exit (0);
}

extern int gettid(void);
/*** keyboard interrupt setup/handler ***/

void kbdint_handler (int sig_num) {
	LOGDEBUG (2, "Intr in thread <%d>\n", mgettid());
	mexit();
}

void setup_kbd_interrupt (void) {
	struct sigaction timer_action;

	timer_action.sa_handler = kbdint_handler;
	timer_action.sa_flags = 0;
	sigemptyset (&timer_action.sa_mask);
	sigaction (SIGINT, & timer_action, NULL);
}

static void sigterm_handler (int sig_num) {
	LOGDEBUG (2, "TERM signal in thread <%d>\n", mgettid());
	mexit();
}

static void setup_daemon_signals (void) {
	struct sigaction timer_action;

	timer_action.sa_handler = sigterm_handler;
	timer_action.sa_flags = 0;
	sigemptyset (&timer_action.sa_mask);
	sigaction (SIGTERM, &timer_action, NULL);
}

/* segmentation fault hanling: dump common structure and dump core  */

static void segv_handler (int sig_num); 
static void setup_segv (void) {
	struct sigaction action;

	action.sa_handler = segv_handler;
	action.sa_flags = 0;
	sigemptyset (&action.sa_mask);
	sigaction (SIGSEGV, & action, NULL);
}

static void segv_handler (int sig_num) {
	struct sigaction action;

	print_core_vars_common (Mcb);
	printf ("restarting\n");

	action.sa_handler = SIG_DFL;
	action.sa_flags = 0;
	sigemptyset (&action.sa_mask);
	sigaction (SIGSEGV, & action, NULL);
}
