//
// Copyright (c) 2020-2021, NVIDIA CORPORATION.  All rights reserved.
//
// NVIDIA CORPORATION and its licensors retain all intellectual property
// and proprietary rights in and to this software, related documentation
// and any modifications thereto.  Any use, reproduction, disclosure or
// distribution of this software and related documentation without an express
// license agreement from NVIDIA CORPORATION is strictly prohibited.
//

#pragma once

#include "nscq/types.hpp"

#include <memory>
#include <vector>

namespace nscq {
    class cqi_session;
} // namespace nscq

namespace nscq::cqi {

auto wait_on_events(std::shared_ptr<nscq::cqi_session>& session_state, uint32_t timeout) -> void;

struct cqi : std::enable_shared_from_this<cqi> {
    using id_t = void;

    // Delete all the default constructors and assignment operators. cqi objects should be managed
    // by smart pointers, so there is no need for copies.
    cqi() = delete;
    cqi(const cqi&) = delete;
    cqi(cqi&&) = delete;
    auto operator=(const cqi&) -> cqi& = delete;
    auto operator=(cqi &&) -> cqi& = delete;

    // Each CQI should have a constructor that takes a parent pointer. This is needed to preserve
    // destructor ordering, e.g., a parent CQI isn't destroyed until the child CQIs are destroyed.
    explicit cqi(std::shared_ptr<cqi> parent) : m_parent(std::move(parent)) {}

    // Virtual destructor to enable derived class destructors to be called from deleting a pointer
    // to this base class.
    virtual ~cqi() = default;

  private:
    std::shared_ptr<cqi> m_parent;
};

template <typename TParent, typename... TBases>
struct field : public TBases... {
    using parent_t = TParent;
    using is_watchable_t = std::false_type;
    using has_set_input_t = std::false_type;

};

template <typename TParent, typename... TBases>
struct session_field : public field<TParent, TBases...> {
  protected:
    std::shared_ptr<nscq::cqi_session>& m_cqi_session;

  public:
    session_field(std::shared_ptr<nscq::cqi_session>& cqi_session) :
        field<TParent, TBases...>(), m_cqi_session(cqi_session) {}

    session_field(std::shared_ptr<nscq::cqi_session>&&) = delete;

    std::shared_ptr<nscq::cqi_session>& get_cqi_session() {
        return m_cqi_session;
    }
};

template <typename TParent, typename... TBases>
struct watchable_field : public session_field<TParent, TBases...> {
  public:
    using is_watchable_t = std::true_type;

    watchable_field(std::shared_ptr<nscq::cqi_session>& cqi_session) :
        session_field<TParent, TBases...>(cqi_session) {}

    watchable_field(std::shared_ptr<nscq::cqi_session>&&) = delete;
};

template <typename TParent, typename... TBases>
struct set_input_field : public session_field<TParent, TBases...> {
  public:
    using has_set_input_t = std::true_type;

    set_input_field(std::shared_ptr<nscq::cqi_session>& cqi_session) :
        session_field<TParent, TBases...>(cqi_session) {}

    set_input_field(std::shared_ptr<nscq::cqi_session>&&) = delete;
};

template <typename TReadValue>
struct readable {
    using read_value_t = TReadValue;
};

template <typename TWriteValue>
struct writable {
    using write_value_t = TWriteValue;
};

} // namespace nscq::cqi
