/*
 * easy_spice - a spice front end
 * Copyright (C) 2001 Routoure Jean-Marc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/*  This part of the code contains the routines for the netlist */
/* the test and so on . */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "easyspice.h"


void 
print_list(GList *list)
{
  for (list=g_list_first(list); list != NULL; list=g_list_next(list)) 
    printf("  %s",(gchar *)list->data);
  printf("\n");
}

void
free_element (void *ptr, gpointer user_data)
{
  g_free(ptr);
}


void
free_keyvalue (KeyValue *kv, gpointer user_data)
{
  g_free(kv->key);
  g_free(kv->value);
  g_free(kv);
}

void
free_device (SpiceDevice *dev, gpointer user_data)
{
  g_list_foreach(dev->result, (GFunc) free_keyvalue, NULL);
  g_list_free(dev->result);
  g_free(dev->name);
}


/* netlist is set NULL if there's a problem */
void 
netlist_read(SpiceNetlist *result ,gchar *filename)
{
  gchar line[FICHIER_MAX_LINE];
  gchar **tokens;
  GList *devices=NULL;
  GList *nodenames=NULL, *tmp_nodenames;
  SpiceDevice *tmp_device;
  
  int i,nodes=0;;
  FILE *fp1;  
    
  if ((fp1=fopen(filename,"rt"))==NULL) {
    write_log(_("\nError: unable to open the netlist file\n"));
    return;
  }

  /* free all memory allocated by result->nodenames and result->devices */
  g_list_foreach(result->nodenames, (GFunc) free_element, NULL);
  g_list_free(result->nodenames);
  result->nodenames=NULL;

  g_list_foreach(result->devices, (GFunc) free_device, NULL);
  g_list_free(result->devices);
  result->devices=NULL;

  while (fgets(line,FICHIER_MAX_LINE,fp1) != NULL) {
    tokens= g_strsplit(line, " ",0);
    g_strdown(tokens[0]);
    tmp_device=g_malloc(sizeof(SpiceDevice));
    tmp_device->name=g_strdup_printf("%s",tokens[0]);
    tmp_device->result=NULL;
    switch (tokens[0][0]) {
    case 'r' : tmp_device->type=SPICE_DIPOLE; nodes=2; break;
    case 'l' : tmp_device->type=SPICE_DIPOLE; nodes=2;break;
    case 'c' : tmp_device->type=SPICE_DIPOLE; nodes=2;break;
    case 'd' : tmp_device->type=SPICE_DIPOLE; nodes=2;break;
    case 'q' : tmp_device->type=SPICE_TRANSISTOR; nodes=3;break;
    case 'w' : tmp_device->type=SPICE_TRANSISTOR; nodes=3;break;
    case 'j' : tmp_device->type=SPICE_TRANSISTOR; nodes=3;break;
    case 'm' : tmp_device->type=SPICE_TRANSISTOR; nodes=4;break;
    case 'e' : tmp_device->type=SPICE_OTHER; nodes=4;break;
    case 'f' : tmp_device->type=SPICE_OTHER; nodes=4;break;
    case 'g' : tmp_device->type=SPICE_OTHER; nodes=4;break;
    case 'h' : tmp_device->type=SPICE_OTHER; nodes=4;break;
    case 't' : tmp_device->type=SPICE_DIPOLE; nodes=4;break;
    case 'o' : tmp_device->type=SPICE_DIPOLE; nodes=4;break;
    case 'v' : tmp_device->type=SPICE_VOLTAGE_SOURCE; nodes=2;break;
    case 'i' : tmp_device->type=SPICE_CURRENT_SOURCE; nodes=2;break;
    case '*' : 
    case '.' :
    case 'x' : 
      tmp_device->name=NULL; 
      if (VERBOSE3)
	printf("Ignore line: \"%s\"\n", tokens[0]); 
      break;
    default : 
      tmp_device->name=NULL; 
      if (VERBOSE3)
	printf("FIXME: unknown device \"%s\"\n",tokens[0]);
    }
    if (tmp_device->name == NULL)
      g_free(tmp_device); /* no device found */
    else {
      if (VERBOSE3)
	printf("  Device: %s \n",tokens[0]);
      for (i=1; i<=nodes; i++) {
	if (VERBOSE3)
	  printf("  Node: %s \n", tokens[i]);
	/* put the node to the list if it is not already in it */
	nodenames=g_list_first(nodenames);
	tmp_nodenames=g_list_find_custom(nodenames, tokens[i], 
				       (GCompareFunc) strcmp);
	if (tmp_nodenames==NULL)
	  nodenames=g_list_insert_sorted(nodenames, g_strdup(tokens[i]),
				       (GCompareFunc) strcmp);
      }
      if (VERBOSE3) {
	print_list(nodenames);
	printf("\n");
      }
      devices=g_list_append(devices, tmp_device);
    }
    g_strfreev(tokens);
  }

  if (DEBUG)
    printf("Close netlist file\n");
  fclose(fp1);
  if (DEBUG)
    printf("Assign results\n");
  result->nodenames=nodenames;
  result->devices=devices;
}


/* ------------------------------------------------------------------ */
/*     Return 0 if any problems                                       */   
/*     Return 1 if an error is found                                  */
/* if affiche==1 then print informations in the log window            */
/* ------------------------------------------------------------------ */

gint 
netlist_test(EasySpice *es, gboolean verbose)
{
  gint return_value, j, gnd=0;
  GList *tmp;
  gchar *node;
  SpiceDevice *dev;
  SpiceNetlist *netlist;
  
  netlist=es->netlist;
  if (DEBUG) 
    printf("Testing the netlist\n");

  /* Test if the *netlist is not empty */
  if (netlist->nodenames==NULL) {
    return_value=1;
    if (DEBUG)
      printf("Netlist is empty\n");
    if (verbose != FALSE)
      write_log(_("No netlist\n"));
  }
  else {
    return_value=0;
    if (DEBUG)
      printf("netlist!=NULL\n");
    if (verbose!=FALSE) {
      clean_log(es); 
      write_log(_("Testing netlist\n")); 
    }

    for (tmp=g_list_first(netlist->nodenames);
	 tmp != NULL; tmp=g_list_next(tmp)) {
      node =(gchar *)tmp->data ;
      if (strcmp(node,"unconnected_pin")==0) {
	return_value=1;
	if (verbose!=FALSE)
	  write_log(_("Error : pin not connected\n"));
      }
      if (strcmp(node,"0")==0)
	gnd++;
    }
    if (gnd==0) {
      return_value=1;
      if (verbose!=FALSE)
	write_log(_("no ground\nMake sure to put an attribute label = 0"
		    " for the ground potential\n"));
    }
    for (tmp=g_list_first(netlist->devices); tmp !=NULL;
	 tmp=g_list_next(tmp)) {
      dev=tmp->data;
      if (strlen(dev->name)==0) {
	return_value=1;
	if (verbose!=FALSE) 
	  write_log(_("One missing uref\n"));
      }
      else
	for (j=0;j<strlen(dev->name);j++) {
	  if ((dev->name)[j]=='?') {
	    return_value=1;
	    if (verbose!=FALSE) {
	      write_log(_("Invalid name for uref:"));
	      write_log(dev->name);
	      write_log("\n");
	    }
	  }
	}
    }
  }
  if ((verbose!=FALSE) && (return_value==1)) 
    write_log(_("Error(s) in the netlist\n"));
  if ((verbose!=FALSE) && (return_value==0))
    write_log(_("No error(s)\n"));
  if (return_value==0)
    led_ok(es->win_main);
  else
    led_false(es->win_main);

  if (DEBUG)
    printf("Leaving netlist_test\n");
  return return_value;
}


/* ------------------------------------------------------------------ */
/* take the netlist and put the informations into window              */
/* ------------------------------------------------------------------ */
void 
netlist2easy(EasySpice *es, SpiceNetlist *netlist)
{
  GList *devices=NULL, *sources=NULL, *v_nodes=NULL;;
  GList *tmp;
  SpiceDevice *dev;
  gchar *entry;
  GtkWidget *win= es->win_main;
  
  /* first, we have to create the list */
  for (tmp=g_list_first(netlist->devices); tmp != NULL;
       tmp=g_list_next(tmp)) {
    dev=tmp->data;
    if (DEBUG)
      printf("%s\n",dev->name);
    devices=g_list_insert_sorted(devices,dev->name, (GCompareFunc) strcmp);
    
    if ((dev->type==SPICE_VOLTAGE_SOURCE )|| (dev->type==SPICE_CURRENT_SOURCE))
      sources=g_list_insert_sorted(sources, dev->name, (GCompareFunc) strcmp);
  }

  /* the node for noise and tf simulation needs to be encapsulated into v(node) */
  for (tmp=g_list_first(netlist->nodenames); tmp != NULL; tmp=g_list_next(tmp)) {
    v_nodes=g_list_append(v_nodes,g_strdup_printf("v(%s)",(char*)(tmp->data)));
  }

  if (VERBOSE3) {
    printf(_("List of devices:\n"));
    print_list(devices);
    printf(_("List of sources:\n"));
    print_list(sources);
  }

  /* second, we have to put the lists into the dropdown menus in the windows */
  /* op analyses */
  /* move the gui setup of the op-section to spice2easy */
  //  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"op_node_combo")),netlist->nodenames);
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"op_device_combo")),devices);

  /* dc analyses */
  entry=g_strdup(get_gtk_entry(win,"dc_source1")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"dc_source1_combo")),sources);
  if (g_list_find_custom(sources,entry,(GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "dc_source1");
  g_free(entry);
      
  entry=g_strdup(get_gtk_entry(win,"dc_source2")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"dc_source2_combo")),sources);
  if (g_list_find_custom(sources,entry,(GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "dc_source2");
  g_free(entry);

  /*  noise analyses */
  entry=g_strdup(get_gtk_entry(win,"noise_source")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"noise_source_combo")),sources);
  if (g_list_find_custom(sources,entry,(GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "noise_source");
  g_free(entry);

  entry=g_strdup(get_gtk_entry(win,"noise_node")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"noise_node_combo")),v_nodes);
  if (g_list_find_custom(v_nodes, entry, (GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "noise_node");
  g_free(entry);

  /* TF analyses */
  entry=g_strdup(get_gtk_entry(win,"tf_output")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"tf_output_combo")),v_nodes);
  if (g_list_find_custom(v_nodes,entry,(GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "tf_output");
  g_free(entry);

  entry=g_strdup(get_gtk_entry(win,"tf_source")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"tf_source_combo")),sources);
  if (g_list_find_custom(sources,entry,(GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "tf_source");
  g_free(entry);

  /* parameter analyses */
  entry=g_strdup(get_gtk_entry(win,"param_name")); /* keep the value if it is in the list */
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(win,"param_name_combo")),devices);
  if (g_list_find_custom(devices, entry, (GCompareFunc) strcmp) != NULL)
    set_gtk_entry(entry, win, "param_name");
  g_free(entry);

  g_list_foreach(v_nodes,(GFunc) free_element, NULL);
  g_list_free(v_nodes);
  g_list_free(devices);
  g_list_free(sources);
}
