/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2009 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: KMCluster.java,v $
 * $Revision: 1.1 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

package com.sun.star.NLPSolver;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import net.adaptivebox.goodness.IGoodnessCompareEngine;
import net.adaptivebox.knowledge.Library;
import net.adaptivebox.knowledge.SearchPoint;
import net.adaptivebox.problem.ProblemEncoder;

/**
 *
 * @author Andreas Schneider <Andreas.Schneider@Sun.COM>
 */
public class KMCluster implements Comparator<SearchPoint> {

    private ProblemEncoder m_problemEncoder;
    private IGoodnessCompareEngine m_comparator;
    private boolean m_maximize;
    private double[] m_center;
    private ArrayList<SearchPoint> m_nodes = new ArrayList<SearchPoint>();

    public ArrayList<SearchPoint> getNodes() {
        return m_nodes;
    }

    public void printCenter() {
        for (int i = 0; i < m_center.length; i++)
            System.out.print(" " + m_center[i]);
    }

    public KMCluster(ProblemEncoder problemEncoder, IGoodnessCompareEngine comparator, boolean maximize) {
        m_problemEncoder = problemEncoder;
        m_comparator = comparator;
        m_maximize = maximize;
        m_center = m_problemEncoder.getEncodedSearchPoint().getLocation();
    }

    public double getDistanceTo(SearchPoint searchPoint) {
        return ArrayMath.getDistance(m_center, searchPoint.getLocation());
    }

    public double updateCenter() {
        double result = 0.0;
        if (m_nodes.size() > 0) {
            double[] newCenter = new double[m_center.length];
            ArrayMath.setVector(newCenter, 0.0);

            for (int k = 0; k < m_nodes.size(); k++)
                ArrayMath.addToVector(newCenter, m_nodes.get(k).getLocation());

            ArrayMath.divideVectorBy(newCenter, newCenter.length);

            result = ArrayMath.getDistance(m_center, newCenter);

            Collections.sort(m_nodes, this);
        } else {
            m_center = m_problemEncoder.getEncodedSearchPoint().getLocation();
        }

        return result;
    }

    public void performNelderMeadStep() {
        if (m_nodes.size() > 2) {
            SearchPoint bestPoint = m_nodes.get(0);
            SearchPoint worstPoint = m_nodes.get(m_nodes.size() - 1);
            double[] centroid = ArrayMath.createVector(m_center.length, 0.0);
            for (int i = 0; i < m_nodes.size() - 1; i++)
                ArrayMath.addToVector(centroid, m_nodes.get(i).getLocation());
            ArrayMath.divideVectorBy(centroid, m_nodes.size() - 1);

            //R = C + (C - W)
            SearchPoint reflectionPoint = m_problemEncoder.getFreshSearchPoint();
            System.arraycopy(centroid, 0, reflectionPoint.getLocation(), 0, centroid.length);
            ArrayMath.subtractFromVector(reflectionPoint.getLocation(), worstPoint.getLocation());
            ArrayMath.addToVector(reflectionPoint.getLocation(), centroid);
            m_problemEncoder.evaluate(reflectionPoint);

            if (m_comparator.compare(reflectionPoint.getEncodeInfo(), bestPoint.getEncodeInfo()) > IGoodnessCompareEngine.EQUAL_TO) {
                //reflection point is better than any other solution
                //we expand further: Re = C + n*(C-W)
                SearchPoint expandedReflectionPoint = m_problemEncoder.getFreshSearchPoint();
                System.arraycopy(centroid, 0, expandedReflectionPoint.getLocation(), 0, centroid.length);
                ArrayMath.subtractFromVector(expandedReflectionPoint.getLocation(), worstPoint.getLocation());
                ArrayMath.multiplyVectorBy(expandedReflectionPoint.getLocation(), 2.0); //TODO : make variable expansion coefficient
                ArrayMath.addToVector(expandedReflectionPoint.getLocation(), centroid);
                m_problemEncoder.evaluate(expandedReflectionPoint);
                worstPoint.importLocation(expandedReflectionPoint);
            } else if (m_comparator.compare(reflectionPoint.getEncodeInfo(), worstPoint.getEncodeInfo()) < IGoodnessCompareEngine.EQUAL_TO) {
                //reflection point is worse than the (up to now) worst point
                //we contract further to the same side of the centroid: Rc = C + k*(C-W)
                SearchPoint contractedReflectionPoint = m_problemEncoder.getFreshSearchPoint();
                System.arraycopy(centroid, 0, contractedReflectionPoint.getLocation(), 0, centroid.length);
                ArrayMath.subtractFromVector(contractedReflectionPoint.getLocation(), worstPoint.getLocation());
                ArrayMath.multiplyVectorBy(contractedReflectionPoint.getLocation(), 0.5); //TODO : make variable contraction coefficient
                ArrayMath.addToVector(contractedReflectionPoint.getLocation(), centroid);
                m_problemEncoder.evaluate(contractedReflectionPoint);
                worstPoint.importLocation(contractedReflectionPoint);
            } else {
                SearchPoint nextWorstPoint = m_nodes.get(m_nodes.size() - 2); //the "not-so-worst" point
                if (m_comparator.compare(reflectionPoint.getEncodeInfo(), nextWorstPoint.getEncodeInfo()) < IGoodnessCompareEngine.EQUAL_TO) {
                    //reflection point is better than the worst point but still worse
                    //than any other point in the solution.
                    //we contract further into the opposite direction: Rc = C - k*(C-W)
                    SearchPoint contractedReflectionPoint = m_problemEncoder.getFreshSearchPoint();
                    System.arraycopy(centroid, 0, contractedReflectionPoint.getLocation(), 0, centroid.length);
                    ArrayMath.subtractFromVector(contractedReflectionPoint.getLocation(), worstPoint.getLocation());
                    ArrayMath.multiplyVectorBy(contractedReflectionPoint.getLocation(), -0.5); //TODO : make variable contraction coefficient
                    ArrayMath.addToVector(contractedReflectionPoint.getLocation(), centroid);
                    m_problemEncoder.evaluate(contractedReflectionPoint);
                    worstPoint.importLocation(contractedReflectionPoint);
                } else {
                    worstPoint.importLocation(reflectionPoint);
                }
            }
            updateCenter();

        }
    }

    //compare two SearchPoints
    public int compare(SearchPoint searchPoint1, SearchPoint searchPoint2) {
        return m_comparator.compare(searchPoint1.getEncodeInfo(), searchPoint2.getEncodeInfo());
    }

}
