//------------------------------------------------------------------------------
// <copyright file="WhitespaceRule.cs" company="Microsoft">
//     Copyright (c) Microsoft Corporation.  All rights reserved.
// </copyright>
// <owner current="true" primary="true">[....]</owner>
// <owner current="false">[....]</owner>
//------------------------------------------------------------------------------

using System.Diagnostics;
using System.IO;
using System.Xml.Xsl.Runtime;

namespace System.Xml.Xsl.Qil {
    /// <summary>
    /// Data structure for storing whitespace rules generated by xsl:strip-space and xsl:preserve-space
    /// </summary>
    internal class WhitespaceRule {
        private string  localName;
        private string  namespaceName;
        private bool    preserveSpace;

        /// <summary>
        /// Allow derived classes to construct empty whitespace rule.
        /// </summary>
        protected WhitespaceRule() {
        }

        /// <summary>
        /// Construct new whitespace rule.
        /// </summary>
        public WhitespaceRule(string localName, string namespaceName, bool preserveSpace) {
            Init(localName, namespaceName, preserveSpace);
        }

        /// <summary>
        /// Initialize whitespace rule after it's been constructed.
        /// </summary>
        protected void Init(string localName, string namespaceName, bool preserveSpace) {
            this.localName = localName;
            this.namespaceName = namespaceName;
            this.preserveSpace = preserveSpace;
        }

        /// <summary>
        /// Local name of the element.
        /// </summary>
        public string LocalName {
            get { return localName; }
            set { localName = value; }
        }

        /// <summary>
        /// Namespace name (uri) of the element.
        /// </summary>
        public string NamespaceName {
            get { return namespaceName; }
            set { namespaceName = value; }
        }

        /// <summary>
        /// True, if this element is whitespace-preserving.
        /// False, if this element is whitespace-stripping.
        /// </summary>
        public bool PreserveSpace {
            get { return preserveSpace; }
        }

        /// <summary>
        /// Serialize the object to BinaryWriter.
        /// </summary>
        public void GetObjectData(XmlQueryDataWriter writer) {
            Debug.Assert(this.GetType() == typeof(WhitespaceRule), "Serialization of WhitespaceRule subclasses is not implemented");
            // string localName;
            writer.WriteStringQ(localName);
            // string namespaceName;
            writer.WriteStringQ(namespaceName);
            // bool preserveSpace;
            writer.Write(preserveSpace);
        }

        /// <summary>
        /// Deserialize the object from BinaryReader.
        /// </summary>
        public WhitespaceRule(XmlQueryDataReader reader) {
            // string localName;
            this.localName = reader.ReadStringQ();
            // string namespaceName;
            this.namespaceName = reader.ReadStringQ();
            // bool preserveSpace;
            this.preserveSpace = reader.ReadBoolean();
        }
    }   
}
