# Copyright 2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Fabric objects."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    "DEFAULT_FABRIC_NAME",
    "Fabric",
    "NAME_VALIDATOR",
    ]

import re

from django.core.exceptions import ValidationError
from django.core.validators import RegexValidator
from django.db.models import CharField
from maasserver import DefaultMeta
from maasserver.models.cleansave import CleanSave
from maasserver.models.timestampedmodel import TimestampedModel


def validate_fabric_name(value):
    """Django validator: `value` must be either `None`, or valid."""
    if value is None:
        return
    namespec = re.compile(r'^[ \w-]+$')
    if not namespec.search(value):
        raise ValidationError("Invalid fabric name: %s." % value)

NAME_VALIDATOR = RegexValidator(r'^[ \w-]+$')

# Name of the special, default fabric.  This fabric cannot be deleted.
DEFAULT_FABRIC_NAME = 'fabric-0'


class Fabric(CleanSave, TimestampedModel):
    """A `Fabric`.

    :ivar name: The short-human-identifiable name for this fabric.
    :ivar objects: An instance of the class :class:`FabricManager`.
    """

    class Meta(DefaultMeta):
        """Needed for South to recognize this model."""
        verbose_name = "Fabric"
        verbose_name_plural = "Fabrics"

    name = CharField(
        max_length=256, editable=True, null=True, blank=True, unique=False,
        validators=[validate_fabric_name])

    class_type = CharField(
        max_length=256, editable=True, null=True, blank=True,
        validators=[NAME_VALIDATOR])
