# Copyright 2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Model for a block devices partition table."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    'PartitionTable',
    ]

from django.db.models import (
    CharField,
    ForeignKey,
)
from maasserver import DefaultMeta
from maasserver.enum import PARTITION_TABLE_TYPE_CHOICES
from maasserver.models.blockdevice import BlockDevice
from maasserver.models.cleansave import CleanSave
from maasserver.models.timestampedmodel import TimestampedModel


# The first 2MiB of the device are used by the partition table and grub. We'll
# reserve the first 4MiB to make sure all partitions stay aligned to 4MB across
# the device.
INITIAL_PARTITION_OFFSET = 4 * 1024 * 1024

# An additional 1MiB of space is left open at the end of the disk to allow for
# the extra MBR table.
END_OF_PARTITION_TABLE_SPACE = 1 * 1024 * 1024

# The amount of extra space a partition table will use.
PARTITION_TABLE_EXTRA_SPACE = (
    INITIAL_PARTITION_OFFSET + END_OF_PARTITION_TABLE_SPACE)


class PartitionTable(CleanSave, TimestampedModel):
    """A partition table on a block device.

    :ivar table_type: Type of partition table.
    :ivar block_device: `BlockDevice` this partition table belongs to.
    """

    class Meta(DefaultMeta):
        """Needed for South to recognize this model."""

    table_type = CharField(
        max_length=20, choices=PARTITION_TABLE_TYPE_CHOICES, default=None)

    block_device = ForeignKey(
        BlockDevice, null=False, blank=False)
