# Copyright 2012-2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""A list of top-level standard Python library names.

This list is used by ``format-imports`` to determine if a module is in this
group or not.

The list is extracted from Python's online documentation, with modules
specific to other OSes taken out. It may need to be updated from time to time.
By default it combines the module lists for Python 2.7 and Python 3.5 so that
reformatting most modern Python source files should DTRT.
"""


# Run this to generate a new module list.
if __name__ == '__main__':
    from distutils.version import StrictVersion
    from operator import methodcaller
    from sys import argv, stdout
    from six.moves.urllib.request import urlopen
    from lxml import html

    versions = argv[1:]
    if len(versions) == 0:
        versions = ["3.5", "2.7"]

    modules = set()

    for version in versions:
        version = StrictVersion(version).version
        modindex_urls = [
            ("http://docs.python.org/release/"
             "{0}.{1}/modindex.html").format(*version),
            ("https://docs.python.org/{0}/"
             "py-modindex.html").format(*version),
        ]
        for modindex_url in modindex_urls:
            try:
                content = urlopen(modindex_url).read()
            except OSError:
                pass
            else:
                root = html.fromstring(content)
                modules.update(
                    node.text.split(".", 1)[0]  # The "base" module name.
                    for node in root.cssselect("table tt"))
                break
        else:
            raise SystemExit(
                "Could not obtain module list from any of the "
                "following URLs: " + " ".join(modindex_urls))

    stdout.write("python_standard_libs = [\n")
    for module in sorted(modules, key=methodcaller("lower")):
        stdout.write("    %r,\n" % module)
    stdout.write("]\n")


python_standard_libs = [
    '__builtin__',
    '__future__',
    '__main__',
    '_dummy_thread',
    '_thread',
    '_winreg',
    'abc',
    'aepack',
    'aetools',
    'aetypes',
    'aifc',
    'AL',
    'al',
    'anydbm',
    'applesingle',
    'argparse',
    'array',
    'ast',
    'asynchat',
    'asyncio',
    'asyncore',
    'atexit',
    'audioop',
    'autoGIL',
    'base64',
    'BaseHTTPServer',
    'Bastion',
    'bdb',
    'binascii',
    'binhex',
    'bisect',
    'bsddb',
    'buildtools',
    'builtins',
    'bz2',
    'calendar',
    'Carbon',
    'cd',
    'cfmfile',
    'cgi',
    'CGIHTTPServer',
    'cgitb',
    'chunk',
    'cmath',
    'cmd',
    'code',
    'codecs',
    'codeop',
    'collections',
    'ColorPicker',
    'colorsys',
    'commands',
    'compileall',
    'compiler',
    'concurrent',
    'ConfigParser',
    'configparser',
    'contextlib',
    'Cookie',
    'cookielib',
    'copy',
    'copy_reg',
    'copyreg',
    'cPickle',
    'cProfile',
    'crypt',
    'cStringIO',
    'csv',
    'ctypes',
    'curses',
    'datetime',
    'dbhash',
    'dbm',
    'decimal',
    'DEVICE',
    'difflib',
    'dircache',
    'dis',
    'distutils',
    'dl',
    'doctest',
    'DocXMLRPCServer',
    'dumbdbm',
    'dummy_thread',
    'dummy_threading',
    'EasyDialogs',
    'email',
    'encodings',
    'ensurepip',
    'enum',
    'errno',
    'exceptions',
    'faulthandler',
    'fcntl',
    'filecmp',
    'fileinput',
    'findertools',
    'fl',
    'FL',
    'flp',
    'fm',
    'fnmatch',
    'formatter',
    'fpectl',
    'fpformat',
    'fractions',
    'FrameWork',
    'ftplib',
    'functools',
    'future_builtins',
    'gc',
    'gdbm',
    'gensuitemodule',
    'getopt',
    'getpass',
    'gettext',
    'GL',
    'gl',
    'glob',
    'grp',
    'gzip',
    'hashlib',
    'heapq',
    'hmac',
    'hotshot',
    'html',
    'htmlentitydefs',
    'htmllib',
    'HTMLParser',
    'http',
    'httplib',
    'ic',
    'icopen',
    'imageop',
    'imaplib',
    'imgfile',
    'imghdr',
    'imp',
    'importlib',
    'imputil',
    'inspect',
    'io',
    'ipaddress',
    'itertools',
    'jpeg',
    'json',
    'keyword',
    'lib2to3',
    'linecache',
    'locale',
    'logging',
    'lzma',
    'macerrors',
    'MacOS',
    'macostools',
    'macpath',
    'macresource',
    'mailbox',
    'mailcap',
    'marshal',
    'math',
    'md5',
    'mhlib',
    'mimetools',
    'mimetypes',
    'MimeWriter',
    'mimify',
    'MiniAEFrame',
    'mmap',
    'modulefinder',
    'msilib',
    'msvcrt',
    'multifile',
    'multiprocessing',
    'mutex',
    'Nav',
    'netrc',
    'new',
    'nis',
    'nntplib',
    'numbers',
    'operator',
    'optparse',
    'os',
    'ossaudiodev',
    'parser',
    'pathlib',
    'pdb',
    'pickle',
    'pickletools',
    'pipes',
    'PixMapWrapper',
    'pkgutil',
    'platform',
    'plistlib',
    'popen2',
    'poplib',
    'posix',
    'posixfile',
    'pprint',
    'profile',
    'pstats',
    'pty',
    'pwd',
    'py_compile',
    'pyclbr',
    'pydoc',
    'queue',
    'Queue',
    'quopri',
    'random',
    're',
    'readline',
    'repr',
    'reprlib',
    'resource',
    'rexec',
    'rfc822',
    'rlcompleter',
    'robotparser',
    'runpy',
    'sched',
    'ScrolledText',
    'select',
    'selectors',
    'sets',
    'sgmllib',
    'sha',
    'shelve',
    'shlex',
    'shutil',
    'signal',
    'SimpleHTTPServer',
    'SimpleXMLRPCServer',
    'site',
    'smtpd',
    'smtplib',
    'sndhdr',
    'socket',
    'socketserver',
    'SocketServer',
    'spwd',
    'sqlite3',
    'ssl',
    'stat',
    'statistics',
    'statvfs',
    'string',
    'StringIO',
    'stringprep',
    'struct',
    'subprocess',
    'sunau',
    'sunaudiodev',
    'SUNAUDIODEV',
    'symbol',
    'symtable',
    'sys',
    'sysconfig',
    'syslog',
    'tabnanny',
    'tarfile',
    'telnetlib',
    'tempfile',
    'termios',
    'test',
    'textwrap',
    'thread',
    'threading',
    'time',
    'timeit',
    'Tix',
    'Tkinter',
    'tkinter',
    'token',
    'tokenize',
    'trace',
    'traceback',
    'tracemalloc',
    'ttk',
    'tty',
    'turtle',
    'turtledemo',
    'types',
    'typing',
    'unicodedata',
    'unittest',
    'urllib',
    'urllib2',
    'urlparse',
    'user',
    'UserDict',
    'UserList',
    'UserString',
    'uu',
    'uuid',
    'venv',
    'videoreader',
    'W',
    'warnings',
    'wave',
    'weakref',
    'webbrowser',
    'whichdb',
    'winreg',
    'winsound',
    'wsgiref',
    'xdrlib',
    'xml',
    'xmlrpc',
    'xmlrpclib',
    'zipapp',
    'zipfile',
    'zipimport',
    'zlib',
]
