/*
 * unity-webapps-indicator-view-messaging-menu.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 *         Lars Uebernickel <lars.uebernickel@canonical.com>
 * 
 * unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include "config.h"
#include <stdlib.h>

#include <messaging-menu.h>

#include "unity-webapps-indicator-view-messaging-menu.h"
#include "unity-webapps-action-manager.h"
#include "unity-webapps-resource-cache.h"

#include "../unity-webapps-debug.h"

G_DEFINE_TYPE(UnityWebappsIndicatorViewMessagingMenu,
              unity_webapps_indicator_view_messaging_menu,
              UNITY_WEBAPPS_TYPE_INDICATOR_VIEW)

enum {
  PROP_0,
  PROP_DESKTOP_FILE
};

struct _UnityWebappsIndicatorViewMessagingMenuPrivate {
  gchar *desktop_file;
  MessagingMenuApp *mmapp;
};

static void
unity_webapps_indicator_view_messaging_menu_source_activated (MessagingMenuApp *mmapp,
                                                              const gchar      *source_id,
                                                              gpointer          user_data)
{
  UnityWebappsIndicatorViewMessagingMenu *view = user_data;

  g_signal_emit_by_name (view, "indicator-raised", source_id);
}

static void
unity_webapps_indicator_view_messaging_menu_create_mmapp (UnityWebappsIndicatorViewMessagingMenu *self)
{
  gchar *desktop_id;

  g_return_if_fail (self->priv->mmapp == NULL);

  /* The messaging menu only accepts desktop file ids and looks for
   * corresponding applications in XDG_DATA_DIRS.  Since we know that
   * priv->desktop_file will always be installed correctly, we can
   * safely extract the file's id (its basename) and pass that to the
   * messaging menu */
  desktop_id = g_path_get_basename (self->priv->desktop_file);

  self->priv->mmapp = messaging_menu_app_new (desktop_id);
  messaging_menu_app_register (self->priv->mmapp);

  g_signal_connect (self->priv->mmapp, "activate-source",
                    G_CALLBACK (unity_webapps_indicator_view_messaging_menu_source_activated),
                    self);

  g_free (desktop_id);
}

static MessagingMenuApp *
unity_webapps_indicator_view_messaging_menu_get_mmapp (UnityWebappsIndicatorViewMessagingMenu *view)
{
  if (view->priv->mmapp != NULL)
    {
      return view->priv->mmapp;
    }
  unity_webapps_indicator_view_messaging_menu_create_mmapp (view);
  return view->priv->mmapp;
}

static void
unity_webapps_indicator_view_messaging_menu_show_indicator (UnityWebappsIndicatorView *indicator_view,
                                                            const gchar *name)
{
  UnityWebappsIndicatorViewMessagingMenu *self = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (indicator_view);
  MessagingMenuApp *mmapp;

  mmapp = unity_webapps_indicator_view_messaging_menu_get_mmapp (self);
  g_return_if_fail (mmapp != NULL);

  if (!messaging_menu_app_has_source (mmapp, name))
    {
      messaging_menu_app_append_source (mmapp, name, NULL, name);
    }
}

static void
unity_webapps_indicator_view_messaging_menu_clear_indicator (UnityWebappsIndicatorView *indicator_view,
                                                             const gchar *name)
{
  UnityWebappsIndicatorViewMessagingMenu *self = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (indicator_view);
  MessagingMenuApp *mmapp;

  mmapp = unity_webapps_indicator_view_messaging_menu_get_mmapp (self);

  g_return_if_fail (mmapp != NULL);

  messaging_menu_app_remove_source (mmapp, name);
}

static void
set_property_icon (MessagingMenuApp *mmapp,
                   const gchar *indicator_name,
                   const gchar *icon_url)
{
  GFile *icon_file;
  gchar *icon_path;
  GIcon *icon;

  icon_path = unity_webapps_resource_cache_lookup_uri (icon_url);

  if (icon_path == NULL)
    {
      UNITY_WEBAPPS_NOTE (INDICATOR, "Resource cache returned NULL for URL (%s)", icon_url);
      return;
    }

  icon_file = g_file_new_for_path (icon_path);

  if (icon_file == NULL)
    {
      UNITY_WEBAPPS_NOTE (INDICATOR, "Failed to construct file from icon path: %s", icon_path);
      return;
    }

  icon = g_file_icon_new (icon_file);

  messaging_menu_app_set_source_icon (mmapp, indicator_name, icon);

  g_object_unref (icon_file);
  g_object_unref (icon);
  g_free (icon_path);
}

static void
unity_webapps_indicator_view_messaging_menu_set_indicator_property (UnityWebappsIndicatorView *indicator_view,
                                                                    const gchar *indicator_name,
                                                                    const gchar *property_name,
                                                                    GVariant *value,
                                                                    gboolean draw_attention)
{
  UnityWebappsIndicatorViewMessagingMenu *self = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (indicator_view);
  MessagingMenuApp *mmapp;

  mmapp = unity_webapps_indicator_view_messaging_menu_get_mmapp (self);
  g_return_if_fail (mmapp != NULL);

  if (g_strcmp0 (property_name, "icon") == 0)
    {
      set_property_icon (mmapp, indicator_name, g_variant_get_string(value, NULL));
    }
  else if (g_strcmp0 (property_name, "label") == 0)
    {
      const gchar *label;

      label = g_variant_get_string (value, NULL);
      messaging_menu_app_set_source_label (mmapp, indicator_name, label);
    }
  else if (g_strcmp0 (property_name, "count") == 0)
    {
      gint count;

      count = atoi (g_variant_get_string (value, NULL));
      messaging_menu_app_set_source_count (mmapp, indicator_name, count);
    }

  if (draw_attention)
    messaging_menu_app_draw_attention (mmapp, indicator_name);
  else
    messaging_menu_app_remove_attention (mmapp, indicator_name);
}

static void
unity_webapps_indicator_view_messaging_menu_finalize (GObject *object)
{
  UnityWebappsIndicatorViewMessagingMenu *view = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (object);

  g_free (view->priv->desktop_file);

  G_OBJECT_CLASS (unity_webapps_indicator_view_messaging_menu_parent_class)->finalize(object);
}


static void
unity_webapps_indicator_view_messaging_menu_dispose (GObject *object)
{
  UnityWebappsIndicatorViewMessagingMenu *view;

  view = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (object);

#if defined(ENABLE_TESTS)
  if (G_IS_OBJECT(view->priv->mmapp)
      && !g_strcmp0(g_getenv("UNITY_WEBAPPS_RUNNING_TRACED_TESTS"), "yes"))
    {
      // Remove apps when we run in the context of traced apps, to avoid
      // over-crowding the messaging menu w/ junk.
      messaging_menu_app_unregister(view->priv->mmapp);
    }
#endif
  g_clear_object (&view->priv->mmapp);

  G_OBJECT_CLASS (unity_webapps_indicator_view_messaging_menu_parent_class)->dispose(object);
}

static void
unity_webapps_indicator_view_messaging_menu_get_property (GObject *object,
                                                          guint prop_id,
                                                          GValue *value,
                                                          GParamSpec *pspec)
{
  UnityWebappsIndicatorViewMessagingMenu *view = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (object);

  switch (prop_id)
    {
    case PROP_DESKTOP_FILE:
      g_value_set_string (value, view->priv->desktop_file);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
unity_webapps_indicator_view_messaging_menu_set_property (GObject *object,
                                                          guint prop_id,
                                                          const GValue *value,
                                                          GParamSpec *pspec)
{
  UnityWebappsIndicatorViewMessagingMenu *view = UNITY_WEBAPPS_INDICATOR_VIEW_MESSAGING_MENU (object);

  switch (prop_id)
    {
    case PROP_DESKTOP_FILE:
      g_return_if_fail (view->priv->desktop_file == NULL);
      view->priv->desktop_file = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
unity_webapps_indicator_view_messaging_menu_class_init (UnityWebappsIndicatorViewMessagingMenuClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  UnityWebappsIndicatorViewClass *indicator_view_class = UNITY_WEBAPPS_INDICATOR_VIEW_CLASS (klass);

  g_type_class_add_private (object_class, sizeof(UnityWebappsIndicatorViewMessagingMenuPrivate));

  object_class->finalize = unity_webapps_indicator_view_messaging_menu_finalize;
  object_class->dispose = unity_webapps_indicator_view_messaging_menu_dispose;

  object_class->get_property = unity_webapps_indicator_view_messaging_menu_get_property;
  object_class->set_property = unity_webapps_indicator_view_messaging_menu_set_property;

  indicator_view_class->show_indicator = unity_webapps_indicator_view_messaging_menu_show_indicator;
  indicator_view_class->clear_indicator = unity_webapps_indicator_view_messaging_menu_clear_indicator;
  indicator_view_class->set_indicator_property = unity_webapps_indicator_view_messaging_menu_set_indicator_property;

  g_object_class_install_property (object_class, PROP_DESKTOP_FILE, g_param_spec_string ("desktop-file", "Desktop File",
                                                                                         "Desktop File to use for the Messaging Menu",
                                                                                         NULL, 
                                                                                         G_PARAM_READWRITE |
                                                                                         G_PARAM_CONSTRUCT_ONLY |
                                                                                         G_PARAM_STATIC_STRINGS));
}

static void
unity_webapps_indicator_view_messaging_menu_init (UnityWebappsIndicatorViewMessagingMenu *view)
{
  view->priv = G_TYPE_INSTANCE_GET_PRIVATE (view,
                                            UNITY_WEBAPPS_TYPE_INDICATOR_VIEW_MESSAGING_MENU,
                                            UnityWebappsIndicatorViewMessagingMenuPrivate);
}

UnityWebappsIndicatorView *
unity_webapps_indicator_view_messaging_menu_new (UnityWebappsIndicatorModel *model, const gchar *desktop_file)
{
  return g_object_new (UNITY_WEBAPPS_TYPE_INDICATOR_VIEW_MESSAGING_MENU,
                       "model", model,
                       "desktop-file", desktop_file,
                       NULL);
}
