// Package keyvault implements the Azure ARM Keyvault service API version
// 2016-10-01.
//
// The key vault client performs cryptographic key operations and vault
// operations against the Key Vault service.
package keyvault

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator 1.0.1.0
// Changes may cause incorrect behavior and will be lost if the code is
// regenerated.

import (
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"net/http"
)

const ()

// ManagementClient is the base client for Keyvault.
type ManagementClient struct {
	autorest.Client
}

// New creates an instance of the ManagementClient client.
func New() ManagementClient {
	return NewWithoutDefaults()
}

// NewWithoutDefaults creates an instance of the ManagementClient client.
func NewWithoutDefaults() ManagementClient {
	return ManagementClient{
		Client: autorest.NewClientWithUserAgent(UserAgent()),
	}
}

// BackupKey the Key Backup operation exports a key from Azure Key Vault in a
// protected form. Note that this operation does NOT return key material in a
// form that can be used outside the Azure Key Vault system, the returned key
// material is either protected to a Azure Key Vault HSM or to Azure Key Vault
// itself. The intent of this operation is to allow a client to GENERATE a key
// in one Azure Key Vault instance, BACKUP the key, and then RESTORE it into
// another Azure Key Vault instance. The BACKUP operation may be used to
// export, in protected form, any key type from Azure Key Vault. Individual
// versions of a key cannot be backed up. BACKUP / RESTORE can be performed
// within geographical boundaries only; meaning that a BACKUP from one
// geographical area cannot be restored to another geographical area. For
// example, a backup from the US geographical area cannot be restored in an EU
// geographical area.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key.
func (client ManagementClient) BackupKey(vaultBaseURL string, keyName string) (result BackupKeyResult, err error) {
	req, err := client.BackupKeyPreparer(vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "BackupKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.BackupKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "BackupKey", resp, "Failure sending request")
		return
	}

	result, err = client.BackupKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "BackupKey", resp, "Failure responding to request")
	}

	return
}

// BackupKeyPreparer prepares the BackupKey request.
func (client ManagementClient) BackupKeyPreparer(vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/backup", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// BackupKeySender sends the BackupKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) BackupKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// BackupKeyResponder handles the response to the BackupKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) BackupKeyResponder(resp *http.Response) (result BackupKeyResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// CreateCertificate if this is the first version, the certificate resource is
// created.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate. parameters is the parameters
// to create a certificate.
func (client ManagementClient) CreateCertificate(vaultBaseURL string, certificateName string, parameters CertificateCreateParameters) (result CertificateOperation, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: certificateName,
			Constraints: []validation.Constraint{{Target: "certificateName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.CertificatePolicy", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
					}},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "CreateCertificate")
	}

	req, err := client.CreateCertificatePreparer(vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "CreateCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "CreateCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.CreateCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "CreateCertificate", resp, "Failure responding to request")
	}

	return
}

// CreateCertificatePreparer prepares the CreateCertificate request.
func (client ManagementClient) CreateCertificatePreparer(vaultBaseURL string, certificateName string, parameters CertificateCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/create", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// CreateCertificateSender sends the CreateCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) CreateCertificateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// CreateCertificateResponder handles the response to the CreateCertificate request. The method always
// closes the http.Response Body.
func (client ManagementClient) CreateCertificateResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// CreateKey the create key operation can be used to create any key type in
// Azure Key Vault. If the named key already exists, Azure Key Vault creates a
// new version of the key.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name for the new key. The system will generate the version
// name for the new key. parameters is the parameters to create a key.
func (client ManagementClient) CreateKey(vaultBaseURL string, keyName string, parameters KeyCreateParameters) (result KeyBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: keyName,
			Constraints: []validation.Constraint{{Target: "keyName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "CreateKey")
	}

	req, err := client.CreateKeyPreparer(vaultBaseURL, keyName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "CreateKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "CreateKey", resp, "Failure sending request")
		return
	}

	result, err = client.CreateKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "CreateKey", resp, "Failure responding to request")
	}

	return
}

// CreateKeyPreparer prepares the CreateKey request.
func (client ManagementClient) CreateKeyPreparer(vaultBaseURL string, keyName string, parameters KeyCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/create", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// CreateKeySender sends the CreateKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) CreateKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// CreateKeyResponder handles the response to the CreateKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) CreateKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Decrypt the DECRYPT operation decrypts a well-formed block of ciphertext
// using the target encryption key and specified algorithm. This operation is
// the reverse of the ENCRYPT operation; only a single block of data may be
// decrypted, the size of this block is dependent on the target key and the
// algorithm to be used. The DECRYPT operation applies to asymmetric and
// symmetric keys stored in Azure Key Vault since it uses the private portion
// of the key.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. keyVersion is the version of the key.
// parameters is the parameters for the decryption operation.
func (client ManagementClient) Decrypt(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "Decrypt")
	}

	req, err := client.DecryptPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Decrypt", nil, "Failure preparing request")
		return
	}

	resp, err := client.DecryptSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Decrypt", resp, "Failure sending request")
		return
	}

	result, err = client.DecryptResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Decrypt", resp, "Failure responding to request")
	}

	return
}

// DecryptPreparer prepares the Decrypt request.
func (client ManagementClient) DecryptPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/decrypt", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DecryptSender sends the Decrypt request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DecryptSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DecryptResponder handles the response to the Decrypt request. The method always
// closes the http.Response Body.
func (client ManagementClient) DecryptResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificate deletes all versions of a certificate object along with
// its associated policy. Delete certificate cannot be used to remove
// individual versions of a certificate object.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate.
func (client ManagementClient) DeleteCertificate(vaultBaseURL string, certificateName string) (result CertificateBundle, err error) {
	req, err := client.DeleteCertificatePreparer(vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificate", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificatePreparer prepares the DeleteCertificate request.
func (client ManagementClient) DeleteCertificatePreparer(vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteCertificateSender sends the DeleteCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteCertificateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteCertificateResponder handles the response to the DeleteCertificate request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateContacts
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
func (client ManagementClient) DeleteCertificateContacts(vaultBaseURL string) (result Contacts, err error) {
	req, err := client.DeleteCertificateContactsPreparer(vaultBaseURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateContacts", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificateContactsPreparer prepares the DeleteCertificateContacts request.
func (client ManagementClient) DeleteCertificateContactsPreparer(vaultBaseURL string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteCertificateContactsSender sends the DeleteCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteCertificateContactsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteCertificateContactsResponder handles the response to the DeleteCertificateContacts request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateIssuer
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// issuerName is the name of the issuer.
func (client ManagementClient) DeleteCertificateIssuer(vaultBaseURL string, issuerName string) (result IssuerBundle, err error) {
	req, err := client.DeleteCertificateIssuerPreparer(vaultBaseURL, issuerName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificateIssuerPreparer prepares the DeleteCertificateIssuer request.
func (client ManagementClient) DeleteCertificateIssuerPreparer(vaultBaseURL string, issuerName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteCertificateIssuerSender sends the DeleteCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteCertificateIssuerResponder handles the response to the DeleteCertificateIssuer request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteCertificateOperation
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate.
func (client ManagementClient) DeleteCertificateOperation(vaultBaseURL string, certificateName string) (result CertificateOperation, err error) {
	req, err := client.DeleteCertificateOperationPreparer(vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteCertificateOperation", resp, "Failure responding to request")
	}

	return
}

// DeleteCertificateOperationPreparer prepares the DeleteCertificateOperation request.
func (client ManagementClient) DeleteCertificateOperationPreparer(vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteCertificateOperationSender sends the DeleteCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteCertificateOperationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteCertificateOperationResponder handles the response to the DeleteCertificateOperation request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteKey the delete key operation cannot be used to remove individual
// versions of a key. This operation removes the cryptographic material
// associated with the key, which means the key is not usable for Sign/Verify,
// Wrap/Unwrap or Encrypt/Decrypt operations.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key to delete.
func (client ManagementClient) DeleteKey(vaultBaseURL string, keyName string) (result KeyBundle, err error) {
	req, err := client.DeleteKeyPreparer(vaultBaseURL, keyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteKey", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteKey", resp, "Failure responding to request")
	}

	return
}

// DeleteKeyPreparer prepares the DeleteKey request.
func (client ManagementClient) DeleteKeyPreparer(vaultBaseURL string, keyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteKeySender sends the DeleteKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteKeyResponder handles the response to the DeleteKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// DeleteSecret the DELETE operation applies to any secret stored in Azure Key
// Vault. DELETE cannot be applied to an individual version of a secret.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// secretName is the name of the secret.
func (client ManagementClient) DeleteSecret(vaultBaseURL string, secretName string) (result SecretBundle, err error) {
	req, err := client.DeleteSecretPreparer(vaultBaseURL, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteSecret", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "DeleteSecret", resp, "Failure responding to request")
	}

	return
}

// DeleteSecretPreparer prepares the DeleteSecret request.
func (client ManagementClient) DeleteSecretPreparer(vaultBaseURL string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteSecretSender sends the DeleteSecret request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) DeleteSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// DeleteSecretResponder handles the response to the DeleteSecret request. The method always
// closes the http.Response Body.
func (client ManagementClient) DeleteSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Encrypt the ENCRYPT operation encrypts an arbitrary sequence of bytes using
// an encryption key that is stored in Azure Key Vault. Note that the ENCRYPT
// operation only supports a single block of data, the size of which is
// dependent on the target key and the encryption algorithm to be used. The
// ENCRYPT operation is only strictly necessary for symmetric keys stored in
// Azure Key Vault since protection with an asymmetric key can be performed
// using public portion of the key. This operation is supported for asymmetric
// keys as a convenience for callers that have a key-reference but do not have
// access to the public key material.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. keyVersion is the version of the key.
// parameters is the parameters for the encryption operation.
func (client ManagementClient) Encrypt(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "Encrypt")
	}

	req, err := client.EncryptPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Encrypt", nil, "Failure preparing request")
		return
	}

	resp, err := client.EncryptSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Encrypt", resp, "Failure sending request")
		return
	}

	result, err = client.EncryptResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Encrypt", resp, "Failure responding to request")
	}

	return
}

// EncryptPreparer prepares the Encrypt request.
func (client ManagementClient) EncryptPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/encrypt", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// EncryptSender sends the Encrypt request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) EncryptSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// EncryptResponder handles the response to the Encrypt request. The method always
// closes the http.Response Body.
func (client ManagementClient) EncryptResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificate the GetCertificate operation returns information about a
// specific certificate in the specified key vault
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate in the given vault.
// certificateVersion is the version of the certificate.
func (client ManagementClient) GetCertificate(vaultBaseURL string, certificateName string, certificateVersion string) (result CertificateBundle, err error) {
	req, err := client.GetCertificatePreparer(vaultBaseURL, certificateName, certificateVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificate", resp, "Failure responding to request")
	}

	return
}

// GetCertificatePreparer prepares the GetCertificate request.
func (client ManagementClient) GetCertificatePreparer(vaultBaseURL string, certificateName string, certificateVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name":    autorest.Encode("path", certificateName),
		"certificate-version": autorest.Encode("path", certificateVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/{certificate-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificateSender sends the GetCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificateResponder handles the response to the GetCertificate request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateContacts the GetCertificateContacts operation returns the set
// of certificate contact resources in the specified key vault.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
func (client ManagementClient) GetCertificateContacts(vaultBaseURL string) (result Contacts, err error) {
	req, err := client.GetCertificateContactsPreparer(vaultBaseURL)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateContacts", resp, "Failure responding to request")
	}

	return
}

// GetCertificateContactsPreparer prepares the GetCertificateContacts request.
func (client ManagementClient) GetCertificateContactsPreparer(vaultBaseURL string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificateContactsSender sends the GetCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificateContactsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificateContactsResponder handles the response to the GetCertificateContacts request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuer the GetCertificateIssuer operation returns the
// specified certificate issuer resources in the specified key vault
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// issuerName is the name of the issuer.
func (client ManagementClient) GetCertificateIssuer(vaultBaseURL string, issuerName string) (result IssuerBundle, err error) {
	req, err := client.GetCertificateIssuerPreparer(vaultBaseURL, issuerName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// GetCertificateIssuerPreparer prepares the GetCertificateIssuer request.
func (client ManagementClient) GetCertificateIssuerPreparer(vaultBaseURL string, issuerName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificateIssuerSender sends the GetCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificateIssuerResponder handles the response to the GetCertificateIssuer request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuers the GetCertificateIssuers operation returns the set of
// certificate issuer resources in the specified key vault
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// maxresults is maximum number of results to return in a page. If not
// specified the service will return up to 25 results.
func (client ManagementClient) GetCertificateIssuers(vaultBaseURL string, maxresults *int32) (result CertificateIssuerListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetCertificateIssuers")
	}

	req, err := client.GetCertificateIssuersPreparer(vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuers", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateIssuersSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuers", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateIssuersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuers", resp, "Failure responding to request")
	}

	return
}

// GetCertificateIssuersPreparer prepares the GetCertificateIssuers request.
func (client ManagementClient) GetCertificateIssuersPreparer(vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/issuers"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificateIssuersSender sends the GetCertificateIssuers request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificateIssuersSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificateIssuersResponder handles the response to the GetCertificateIssuers request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificateIssuersResponder(resp *http.Response) (result CertificateIssuerListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateIssuersNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetCertificateIssuersNextResults(lastResults CertificateIssuerListResult) (result CertificateIssuerListResult, err error) {
	req, err := lastResults.CertificateIssuerListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuers", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetCertificateIssuersSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuers", resp, "Failure sending next results request")
	}

	result, err = client.GetCertificateIssuersResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateIssuers", resp, "Failure responding to next results request")
	}

	return
}

// GetCertificateOperation the GetCertificateOperation operation returns the
// certificate operation associated with the certificate.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate.
func (client ManagementClient) GetCertificateOperation(vaultBaseURL string, certificateName string) (result CertificateOperation, err error) {
	req, err := client.GetCertificateOperationPreparer(vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateOperation", resp, "Failure responding to request")
	}

	return
}

// GetCertificateOperationPreparer prepares the GetCertificateOperation request.
func (client ManagementClient) GetCertificateOperationPreparer(vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificateOperationSender sends the GetCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificateOperationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificateOperationResponder handles the response to the GetCertificateOperation request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificatePolicy the GetCertificatePolicy operation returns the
// specified certificate policy resources in the specified key vault
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate in a given key vault.
func (client ManagementClient) GetCertificatePolicy(vaultBaseURL string, certificateName string) (result CertificatePolicy, err error) {
	req, err := client.GetCertificatePolicyPreparer(vaultBaseURL, certificateName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificatePolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificatePolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificatePolicy", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificatePolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificatePolicy", resp, "Failure responding to request")
	}

	return
}

// GetCertificatePolicyPreparer prepares the GetCertificatePolicy request.
func (client ManagementClient) GetCertificatePolicyPreparer(vaultBaseURL string, certificateName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/policy", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificatePolicySender sends the GetCertificatePolicy request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificatePolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificatePolicyResponder handles the response to the GetCertificatePolicy request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificatePolicyResponder(resp *http.Response) (result CertificatePolicy, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificates the GetCertificates operation returns the set of
// certificates resources in the specified key vault.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// maxresults is maximum number of results to return in a page. If not
// specified the service will return up to 25 results.
func (client ManagementClient) GetCertificates(vaultBaseURL string, maxresults *int32) (result CertificateListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetCertificates")
	}

	req, err := client.GetCertificatesPreparer(vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificates", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificatesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificates", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificates", resp, "Failure responding to request")
	}

	return
}

// GetCertificatesPreparer prepares the GetCertificates request.
func (client ManagementClient) GetCertificatesPreparer(vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificatesSender sends the GetCertificates request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificatesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificatesResponder handles the response to the GetCertificates request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificatesResponder(resp *http.Response) (result CertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificatesNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetCertificatesNextResults(lastResults CertificateListResult) (result CertificateListResult, err error) {
	req, err := lastResults.CertificateListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificates", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetCertificatesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificates", resp, "Failure sending next results request")
	}

	result, err = client.GetCertificatesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificates", resp, "Failure responding to next results request")
	}

	return
}

// GetCertificateVersions the GetCertificateVersions operation returns the
// versions of a certificate in the specified key vault
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate. maxresults is maximum number
// of results to return in a page. If not specified the service will return up
// to 25 results.
func (client ManagementClient) GetCertificateVersions(vaultBaseURL string, certificateName string, maxresults *int32) (result CertificateListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetCertificateVersions")
	}

	req, err := client.GetCertificateVersionsPreparer(vaultBaseURL, certificateName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCertificateVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateVersions", resp, "Failure sending request")
		return
	}

	result, err = client.GetCertificateVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateVersions", resp, "Failure responding to request")
	}

	return
}

// GetCertificateVersionsPreparer prepares the GetCertificateVersions request.
func (client ManagementClient) GetCertificateVersionsPreparer(vaultBaseURL string, certificateName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetCertificateVersionsSender sends the GetCertificateVersions request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetCertificateVersionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetCertificateVersionsResponder handles the response to the GetCertificateVersions request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetCertificateVersionsResponder(resp *http.Response) (result CertificateListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCertificateVersionsNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetCertificateVersionsNextResults(lastResults CertificateListResult) (result CertificateListResult, err error) {
	req, err := lastResults.CertificateListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateVersions", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetCertificateVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateVersions", resp, "Failure sending next results request")
	}

	result, err = client.GetCertificateVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetCertificateVersions", resp, "Failure responding to next results request")
	}

	return
}

// GetKey the get key operation is applicable to all key types. If the
// requested key is symmetric, then no key material is released in the
// response.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key to get. keyVersion is adding the version
// parameter retrieves a specific version of a key.
func (client ManagementClient) GetKey(vaultBaseURL string, keyName string, keyVersion string) (result KeyBundle, err error) {
	req, err := client.GetKeyPreparer(vaultBaseURL, keyName, keyVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKey", resp, "Failure sending request")
		return
	}

	result, err = client.GetKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKey", resp, "Failure responding to request")
	}

	return
}

// GetKeyPreparer prepares the GetKey request.
func (client ManagementClient) GetKeyPreparer(vaultBaseURL string, keyName string, keyVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetKeySender sends the GetKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetKeyResponder handles the response to the GetKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetKeys retrieves a list of the keys in the Key Vault as JSON Web Key
// structures that contain the public part of a stored key. The LIST operation
// is applicable to all key types, however only the base key
// identifier,attributes, and tags are provided in the response. Individual
// versions of a key are not listed in the response. Authorization: Requires
// the keys/list permission.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// maxresults is maximum number of results to return in a page. If not
// specified the service will return up to 25 results.
func (client ManagementClient) GetKeys(vaultBaseURL string, maxresults *int32) (result KeyListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetKeys")
	}

	req, err := client.GetKeysPreparer(vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeys", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeysSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeys", resp, "Failure sending request")
		return
	}

	result, err = client.GetKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeys", resp, "Failure responding to request")
	}

	return
}

// GetKeysPreparer prepares the GetKeys request.
func (client ManagementClient) GetKeysPreparer(vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/keys"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetKeysSender sends the GetKeys request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetKeysSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetKeysResponder handles the response to the GetKeys request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetKeysResponder(resp *http.Response) (result KeyListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetKeysNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetKeysNextResults(lastResults KeyListResult) (result KeyListResult, err error) {
	req, err := lastResults.KeyListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeys", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetKeysSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeys", resp, "Failure sending next results request")
	}

	result, err = client.GetKeysResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeys", resp, "Failure responding to next results request")
	}

	return
}

// GetKeyVersions the full key identifier, attributes, and tags are provided in
// the response.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. maxresults is maximum number of results to
// return in a page. If not specified the service will return up to 25 results.
func (client ManagementClient) GetKeyVersions(vaultBaseURL string, keyName string, maxresults *int32) (result KeyListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetKeyVersions")
	}

	req, err := client.GetKeyVersionsPreparer(vaultBaseURL, keyName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeyVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetKeyVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeyVersions", resp, "Failure sending request")
		return
	}

	result, err = client.GetKeyVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeyVersions", resp, "Failure responding to request")
	}

	return
}

// GetKeyVersionsPreparer prepares the GetKeyVersions request.
func (client ManagementClient) GetKeyVersionsPreparer(vaultBaseURL string, keyName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetKeyVersionsSender sends the GetKeyVersions request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetKeyVersionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetKeyVersionsResponder handles the response to the GetKeyVersions request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetKeyVersionsResponder(resp *http.Response) (result KeyListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetKeyVersionsNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetKeyVersionsNextResults(lastResults KeyListResult) (result KeyListResult, err error) {
	req, err := lastResults.KeyListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeyVersions", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetKeyVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeyVersions", resp, "Failure sending next results request")
	}

	result, err = client.GetKeyVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetKeyVersions", resp, "Failure responding to next results request")
	}

	return
}

// GetSecret the GET operation is applicable to any secret stored in Azure Key
// Vault.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// secretName is the name of the secret. secretVersion is the version of the
// secret.
func (client ManagementClient) GetSecret(vaultBaseURL string, secretName string, secretVersion string) (result SecretBundle, err error) {
	req, err := client.GetSecretPreparer(vaultBaseURL, secretName, secretVersion)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.GetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecret", resp, "Failure responding to request")
	}

	return
}

// GetSecretPreparer prepares the GetSecret request.
func (client ManagementClient) GetSecretPreparer(vaultBaseURL string, secretName string, secretVersion string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name":    autorest.Encode("path", secretName),
		"secret-version": autorest.Encode("path", secretVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/{secret-version}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetSecretSender sends the GetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetSecretResponder handles the response to the GetSecret request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSecrets the LIST operation is applicable to the entire vault, however
// only the base secret identifier and attributes are provided in the response.
// Individual secret versions are not listed in the response.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// maxresults is maximum number of results to return in a page. If not
// specified the service will return up to 25 results.
func (client ManagementClient) GetSecrets(vaultBaseURL string, maxresults *int32) (result SecretListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetSecrets")
	}

	req, err := client.GetSecretsPreparer(vaultBaseURL, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecrets", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecrets", resp, "Failure sending request")
		return
	}

	result, err = client.GetSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecrets", resp, "Failure responding to request")
	}

	return
}

// GetSecretsPreparer prepares the GetSecrets request.
func (client ManagementClient) GetSecretsPreparer(vaultBaseURL string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/secrets"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetSecretsSender sends the GetSecrets request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetSecretsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetSecretsResponder handles the response to the GetSecrets request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetSecretsResponder(resp *http.Response) (result SecretListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSecretsNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetSecretsNextResults(lastResults SecretListResult) (result SecretListResult, err error) {
	req, err := lastResults.SecretListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecrets", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetSecretsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecrets", resp, "Failure sending next results request")
	}

	result, err = client.GetSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecrets", resp, "Failure responding to next results request")
	}

	return
}

// GetSecretVersions the LIST VERSIONS operation can be applied to all versions
// having the same secret name in the same key vault. The full secret
// identifier and attributes are provided in the response. No values are
// returned for the secrets and only current versions of a secret are listed.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// secretName is the name of the secret. maxresults is maximum number of
// results to return in a page. If not specified the service will return up to
// 25 results.
func (client ManagementClient) GetSecretVersions(vaultBaseURL string, secretName string, maxresults *int32) (result SecretListResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxresults,
			Constraints: []validation.Constraint{{Target: "maxresults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxresults", Name: validation.InclusiveMaximum, Rule: 25, Chain: nil},
					{Target: "maxresults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "GetSecretVersions")
	}

	req, err := client.GetSecretVersionsPreparer(vaultBaseURL, secretName, maxresults)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecretVersions", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecretVersions", resp, "Failure sending request")
		return
	}

	result, err = client.GetSecretVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecretVersions", resp, "Failure responding to request")
	}

	return
}

// GetSecretVersionsPreparer prepares the GetSecretVersions request.
func (client ManagementClient) GetSecretVersionsPreparer(vaultBaseURL string, secretName string, maxresults *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if maxresults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxresults)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/versions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetSecretVersionsSender sends the GetSecretVersions request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) GetSecretVersionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// GetSecretVersionsResponder handles the response to the GetSecretVersions request. The method always
// closes the http.Response Body.
func (client ManagementClient) GetSecretVersionsResponder(resp *http.Response) (result SecretListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSecretVersionsNextResults retrieves the next set of results, if any.
func (client ManagementClient) GetSecretVersionsNextResults(lastResults SecretListResult) (result SecretListResult, err error) {
	req, err := lastResults.SecretListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecretVersions", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}

	resp, err := client.GetSecretVersionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecretVersions", resp, "Failure sending next results request")
	}

	result, err = client.GetSecretVersionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "GetSecretVersions", resp, "Failure responding to next results request")
	}

	return
}

// ImportCertificate imports an existing valid certificate, containing a
// private key, into Azure Key Vault. The certificate to be imported can be in
// either PFX or PEM format. If the certificate is in PEM format the PEM file
// must contain the key as well as x509 certificates.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate. parameters is the parameters
// to import the certificate.
func (client ManagementClient) ImportCertificate(vaultBaseURL string, certificateName string, parameters CertificateImportParameters) (result CertificateBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: certificateName,
			Constraints: []validation.Constraint{{Target: "certificateName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Base64EncodedCertificate", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.CertificatePolicy", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties", Name: validation.Null, Rule: false,
						Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.Null, Rule: false,
							Chain: []validation.Constraint{{Target: "parameters.CertificatePolicy.X509CertificateProperties.ValidityInMonths", Name: validation.InclusiveMinimum, Rule: 0, Chain: nil}}},
						}},
					}}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "ImportCertificate")
	}

	req, err := client.ImportCertificatePreparer(vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "ImportCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.ImportCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "ImportCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.ImportCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "ImportCertificate", resp, "Failure responding to request")
	}

	return
}

// ImportCertificatePreparer prepares the ImportCertificate request.
func (client ManagementClient) ImportCertificatePreparer(vaultBaseURL string, certificateName string, parameters CertificateImportParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/import", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ImportCertificateSender sends the ImportCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) ImportCertificateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// ImportCertificateResponder handles the response to the ImportCertificate request. The method always
// closes the http.Response Body.
func (client ManagementClient) ImportCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ImportKey the import key operation may be used to import any key type into
// an Azure Key Vault. If the named key already exists, Azure Key Vault creates
// a new version of the key.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is name for the imported key. parameters is the parameters to import
// a key.
func (client ManagementClient) ImportKey(vaultBaseURL string, keyName string, parameters KeyImportParameters) (result KeyBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: keyName,
			Constraints: []validation.Constraint{{Target: "keyName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Key", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "ImportKey")
	}

	req, err := client.ImportKeyPreparer(vaultBaseURL, keyName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "ImportKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.ImportKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "ImportKey", resp, "Failure sending request")
		return
	}

	result, err = client.ImportKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "ImportKey", resp, "Failure responding to request")
	}

	return
}

// ImportKeyPreparer prepares the ImportKey request.
func (client ManagementClient) ImportKeyPreparer(vaultBaseURL string, keyName string, parameters KeyImportParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name": autorest.Encode("path", keyName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ImportKeySender sends the ImportKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) ImportKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// ImportKeyResponder handles the response to the ImportKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) ImportKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// MergeCertificate
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate. parameters is the parameters
// to merge certificate.
func (client ManagementClient) MergeCertificate(vaultBaseURL string, certificateName string, parameters CertificateMergeParameters) (result CertificateBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.X509Certificates", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "MergeCertificate")
	}

	req, err := client.MergeCertificatePreparer(vaultBaseURL, certificateName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "MergeCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.MergeCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "MergeCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.MergeCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "MergeCertificate", resp, "Failure responding to request")
	}

	return
}

// MergeCertificatePreparer prepares the MergeCertificate request.
func (client ManagementClient) MergeCertificatePreparer(vaultBaseURL string, certificateName string, parameters CertificateMergeParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending/merge", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// MergeCertificateSender sends the MergeCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) MergeCertificateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// MergeCertificateResponder handles the response to the MergeCertificate request. The method always
// closes the http.Response Body.
func (client ManagementClient) MergeCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// RestoreKey imports a previously backed up key into Azure Key Vault,
// restoring the key, its key identifier, attributes and access control
// policies. The RESTORE operation may be used to import a previously backed up
// key. Individual versions of a key cannot be restored. The key is restored in
// its entirety with the same key name as it had when it was backed up. If the
// key name is not available in the target Key Vault, the RESTORE operation
// will be rejected. While the key name is retained during restore, the final
// key identifier will change if the key is restored to a different vault.
// Restore will restore all versions and preserve version identifiers. The
// RESTORE operation is subject to security constraints: The target Key Vault
// must be owned by the same Microsoft Azure Subscription as the source Key
// Vault The user must have RESTORE permission in the target Key Vault.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// parameters is the parameters to restore the key.
func (client ManagementClient) RestoreKey(vaultBaseURL string, parameters KeyRestoreParameters) (result KeyBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.KeyBundleBackup", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "RestoreKey")
	}

	req, err := client.RestoreKeyPreparer(vaultBaseURL, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "RestoreKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.RestoreKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "RestoreKey", resp, "Failure sending request")
		return
	}

	result, err = client.RestoreKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "RestoreKey", resp, "Failure responding to request")
	}

	return
}

// RestoreKeyPreparer prepares the RestoreKey request.
func (client ManagementClient) RestoreKeyPreparer(vaultBaseURL string, parameters KeyRestoreParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/keys/restore"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RestoreKeySender sends the RestoreKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) RestoreKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// RestoreKeyResponder handles the response to the RestoreKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) RestoreKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetCertificateContacts
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// contacts is the contacts for the key vault certificate.
func (client ManagementClient) SetCertificateContacts(vaultBaseURL string, contacts Contacts) (result Contacts, err error) {
	req, err := client.SetCertificateContactsPreparer(vaultBaseURL, contacts)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetCertificateContacts", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetCertificateContactsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetCertificateContacts", resp, "Failure sending request")
		return
	}

	result, err = client.SetCertificateContactsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetCertificateContacts", resp, "Failure responding to request")
	}

	return
}

// SetCertificateContactsPreparer prepares the SetCertificateContacts request.
func (client ManagementClient) SetCertificateContactsPreparer(vaultBaseURL string, contacts Contacts) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPath("/certificates/contacts"),
		autorest.WithJSON(contacts),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetCertificateContactsSender sends the SetCertificateContacts request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SetCertificateContactsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SetCertificateContactsResponder handles the response to the SetCertificateContacts request. The method always
// closes the http.Response Body.
func (client ManagementClient) SetCertificateContactsResponder(resp *http.Response) (result Contacts, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetCertificateIssuer
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// issuerName is the name of the issuer. parameter is certificate issuer set
// parameter.
func (client ManagementClient) SetCertificateIssuer(vaultBaseURL string, issuerName string, parameter CertificateIssuerSetParameters) (result IssuerBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameter,
			Constraints: []validation.Constraint{{Target: "parameter.Provider", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "SetCertificateIssuer")
	}

	req, err := client.SetCertificateIssuerPreparer(vaultBaseURL, issuerName, parameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.SetCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// SetCertificateIssuerPreparer prepares the SetCertificateIssuer request.
func (client ManagementClient) SetCertificateIssuerPreparer(vaultBaseURL string, issuerName string, parameter CertificateIssuerSetParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithJSON(parameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetCertificateIssuerSender sends the SetCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SetCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SetCertificateIssuerResponder handles the response to the SetCertificateIssuer request. The method always
// closes the http.Response Body.
func (client ManagementClient) SetCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetSecret the SET operation adds a secret to the Azure Key Vault. If the
// named secret already exists, Azure Key Vault creates a new version of that
// secret.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// secretName is the name of the secret. parameters is the parameters for
// setting the secret.
func (client ManagementClient) SetSecret(vaultBaseURL string, secretName string, parameters SecretSetParameters) (result SecretBundle, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: secretName,
			Constraints: []validation.Constraint{{Target: "secretName", Name: validation.Pattern, Rule: `^[0-9a-zA-Z-]+$`, Chain: nil}}},
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "SetSecret")
	}

	req, err := client.SetSecretPreparer(vaultBaseURL, secretName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.SetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "SetSecret", resp, "Failure responding to request")
	}

	return
}

// SetSecretPreparer prepares the SetSecret request.
func (client ManagementClient) SetSecretPreparer(vaultBaseURL string, secretName string, parameters SecretSetParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name": autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetSecretSender sends the SetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SetSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SetSecretResponder handles the response to the SetSecret request. The method always
// closes the http.Response Body.
func (client ManagementClient) SetSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Sign the SIGN operation is applicable to asymmetric and symmetric keys
// stored in Azure Key Vault since this operation uses the private portion of
// the key.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. keyVersion is the version of the key.
// parameters is the parameters for the signing operation.
func (client ManagementClient) Sign(vaultBaseURL string, keyName string, keyVersion string, parameters KeySignParameters) (result KeyOperationResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "Sign")
	}

	req, err := client.SignPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Sign", nil, "Failure preparing request")
		return
	}

	resp, err := client.SignSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Sign", resp, "Failure sending request")
		return
	}

	result, err = client.SignResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Sign", resp, "Failure responding to request")
	}

	return
}

// SignPreparer prepares the Sign request.
func (client ManagementClient) SignPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeySignParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/sign", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SignSender sends the Sign request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) SignSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// SignResponder handles the response to the Sign request. The method always
// closes the http.Response Body.
func (client ManagementClient) SignResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UnwrapKey the UNWRAP operation supports decryption of a symmetric key using
// the target key encryption key. This operation is the reverse of the WRAP
// operation. The UNWRAP operation applies to asymmetric and symmetric keys
// stored in Azure Key Vault since it uses the private portion of the key.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. keyVersion is the version of the key.
// parameters is the parameters for the key operation.
func (client ManagementClient) UnwrapKey(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "UnwrapKey")
	}

	req, err := client.UnwrapKeyPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UnwrapKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.UnwrapKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UnwrapKey", resp, "Failure sending request")
		return
	}

	result, err = client.UnwrapKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UnwrapKey", resp, "Failure responding to request")
	}

	return
}

// UnwrapKeyPreparer prepares the UnwrapKey request.
func (client ManagementClient) UnwrapKeyPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/unwrapkey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UnwrapKeySender sends the UnwrapKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UnwrapKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UnwrapKeyResponder handles the response to the UnwrapKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) UnwrapKeyResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificate
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate in the given key vault.
// certificateVersion is the version of the certificate. parameters is the
// parameters for certificate update.
func (client ManagementClient) UpdateCertificate(vaultBaseURL string, certificateName string, certificateVersion string, parameters CertificateUpdateParameters) (result CertificateBundle, err error) {
	req, err := client.UpdateCertificatePreparer(vaultBaseURL, certificateName, certificateVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificate", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificate", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificate", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificatePreparer prepares the UpdateCertificate request.
func (client ManagementClient) UpdateCertificatePreparer(vaultBaseURL string, certificateName string, certificateVersion string, parameters CertificateUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name":    autorest.Encode("path", certificateName),
		"certificate-version": autorest.Encode("path", certificateVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/{certificate-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UpdateCertificateSender sends the UpdateCertificate request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UpdateCertificateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UpdateCertificateResponder handles the response to the UpdateCertificate request. The method always
// closes the http.Response Body.
func (client ManagementClient) UpdateCertificateResponder(resp *http.Response) (result CertificateBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificateIssuer
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// issuerName is the name of the issuer. parameter is certificate issuer update
// parameter.
func (client ManagementClient) UpdateCertificateIssuer(vaultBaseURL string, issuerName string, parameter CertificateIssuerUpdateParameters) (result IssuerBundle, err error) {
	req, err := client.UpdateCertificateIssuerPreparer(vaultBaseURL, issuerName, parameter)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificateIssuer", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateIssuerSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificateIssuer", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateIssuerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificateIssuer", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificateIssuerPreparer prepares the UpdateCertificateIssuer request.
func (client ManagementClient) UpdateCertificateIssuerPreparer(vaultBaseURL string, issuerName string, parameter CertificateIssuerUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"issuer-name": autorest.Encode("path", issuerName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/issuers/{issuer-name}", pathParameters),
		autorest.WithJSON(parameter),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UpdateCertificateIssuerSender sends the UpdateCertificateIssuer request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UpdateCertificateIssuerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UpdateCertificateIssuerResponder handles the response to the UpdateCertificateIssuer request. The method always
// closes the http.Response Body.
func (client ManagementClient) UpdateCertificateIssuerResponder(resp *http.Response) (result IssuerBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificateOperation
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate. certificateOperation is the
// certificate operation response.
func (client ManagementClient) UpdateCertificateOperation(vaultBaseURL string, certificateName string, certificateOperation CertificateOperationUpdateParameter) (result CertificateOperation, err error) {
	req, err := client.UpdateCertificateOperationPreparer(vaultBaseURL, certificateName, certificateOperation)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificateOperation", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificateOperationSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificateOperation", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificateOperationResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificateOperation", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificateOperationPreparer prepares the UpdateCertificateOperation request.
func (client ManagementClient) UpdateCertificateOperationPreparer(vaultBaseURL string, certificateName string, certificateOperation CertificateOperationUpdateParameter) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/pending", pathParameters),
		autorest.WithJSON(certificateOperation),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UpdateCertificateOperationSender sends the UpdateCertificateOperation request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UpdateCertificateOperationSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UpdateCertificateOperationResponder handles the response to the UpdateCertificateOperation request. The method always
// closes the http.Response Body.
func (client ManagementClient) UpdateCertificateOperationResponder(resp *http.Response) (result CertificateOperation, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateCertificatePolicy set specified members in the certificate policy.
// Leave others as null.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// certificateName is the name of the certificate in the given vault.
// certificatePolicy is the policy for the certificate.
func (client ManagementClient) UpdateCertificatePolicy(vaultBaseURL string, certificateName string, certificatePolicy CertificatePolicy) (result CertificatePolicy, err error) {
	req, err := client.UpdateCertificatePolicyPreparer(vaultBaseURL, certificateName, certificatePolicy)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificatePolicy", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCertificatePolicySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificatePolicy", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCertificatePolicyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateCertificatePolicy", resp, "Failure responding to request")
	}

	return
}

// UpdateCertificatePolicyPreparer prepares the UpdateCertificatePolicy request.
func (client ManagementClient) UpdateCertificatePolicyPreparer(vaultBaseURL string, certificateName string, certificatePolicy CertificatePolicy) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"certificate-name": autorest.Encode("path", certificateName),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/certificates/{certificate-name}/policy", pathParameters),
		autorest.WithJSON(certificatePolicy),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UpdateCertificatePolicySender sends the UpdateCertificatePolicy request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UpdateCertificatePolicySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UpdateCertificatePolicyResponder handles the response to the UpdateCertificatePolicy request. The method always
// closes the http.Response Body.
func (client ManagementClient) UpdateCertificatePolicyResponder(resp *http.Response) (result CertificatePolicy, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateKey in order to perform this operation, the key must already exist in
// the Key Vault. Note: The cryptographic material of a key itself cannot be
// changed.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of key to update. keyVersion is the version of the key
// to update. parameters is the parameters of the key to update.
func (client ManagementClient) UpdateKey(vaultBaseURL string, keyName string, keyVersion string, parameters KeyUpdateParameters) (result KeyBundle, err error) {
	req, err := client.UpdateKeyPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateKey", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateKey", resp, "Failure responding to request")
	}

	return
}

// UpdateKeyPreparer prepares the UpdateKey request.
func (client ManagementClient) UpdateKeyPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeyUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UpdateKeySender sends the UpdateKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UpdateKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UpdateKeyResponder handles the response to the UpdateKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) UpdateKeyResponder(resp *http.Response) (result KeyBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// UpdateSecret the UPDATE operation changes specified attributes of an
// existing stored secret. Attributes that are not specified in the request are
// left unchanged. The value of a secret itself cannot be changed.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// secretName is the name of the secret. secretVersion is the version of the
// secret. parameters is the parameters for update secret operation.
func (client ManagementClient) UpdateSecret(vaultBaseURL string, secretName string, secretVersion string, parameters SecretUpdateParameters) (result SecretBundle, err error) {
	req, err := client.UpdateSecretPreparer(vaultBaseURL, secretName, secretVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateSecret", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "UpdateSecret", resp, "Failure responding to request")
	}

	return
}

// UpdateSecretPreparer prepares the UpdateSecret request.
func (client ManagementClient) UpdateSecretPreparer(vaultBaseURL string, secretName string, secretVersion string, parameters SecretUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"secret-name":    autorest.Encode("path", secretName),
		"secret-version": autorest.Encode("path", secretVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/secrets/{secret-name}/{secret-version}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// UpdateSecretSender sends the UpdateSecret request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) UpdateSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// UpdateSecretResponder handles the response to the UpdateSecret request. The method always
// closes the http.Response Body.
func (client ManagementClient) UpdateSecretResponder(resp *http.Response) (result SecretBundle, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Verify the VERIFY operation is applicable to symmetric keys stored in Azure
// Key Vault. VERIFY is not strictly necessary for asymmetric keys stored in
// Azure Key Vault since signature verification can be performed using the
// public portion of the key but this operation is supported as a convenience
// for callers that only have a key-reference and not the public portion of the
// key.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. keyVersion is the version of the key.
// parameters is the parameters for verify operations.
func (client ManagementClient) Verify(vaultBaseURL string, keyName string, keyVersion string, parameters KeyVerifyParameters) (result KeyVerifyResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Digest", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.Signature", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "Verify")
	}

	req, err := client.VerifyPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Verify", nil, "Failure preparing request")
		return
	}

	resp, err := client.VerifySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Verify", resp, "Failure sending request")
		return
	}

	result, err = client.VerifyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "Verify", resp, "Failure responding to request")
	}

	return
}

// VerifyPreparer prepares the Verify request.
func (client ManagementClient) VerifyPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeyVerifyParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/verify", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// VerifySender sends the Verify request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) VerifySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// VerifyResponder handles the response to the Verify request. The method always
// closes the http.Response Body.
func (client ManagementClient) VerifyResponder(resp *http.Response) (result KeyVerifyResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// WrapKey the WRAP operation supports encryption of a symmetric key using a
// key encryption key that has previously been stored in an Azure Key Vault.
// The WRAP operation is only strictly necessary for symmetric keys stored in
// Azure Key Vault since protection with an asymmetric key can be performed
// using the public portion of the key. This operation is supported for
// asymmetric keys as a convenience for callers that have a key-reference but
// do not have access to the public key material.
//
// vaultBaseURL is the vault name, for example https://myvault.vault.azure.net.
// keyName is the name of the key. keyVersion is the version of the key.
// parameters is the parameters for wrap operation.
func (client ManagementClient) WrapKey(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (result KeyOperationResult, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Value", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "keyvault.ManagementClient", "WrapKey")
	}

	req, err := client.WrapKeyPreparer(vaultBaseURL, keyName, keyVersion, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "WrapKey", nil, "Failure preparing request")
		return
	}

	resp, err := client.WrapKeySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "WrapKey", resp, "Failure sending request")
		return
	}

	result, err = client.WrapKeyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "keyvault.ManagementClient", "WrapKey", resp, "Failure responding to request")
	}

	return
}

// WrapKeyPreparer prepares the WrapKey request.
func (client ManagementClient) WrapKeyPreparer(vaultBaseURL string, keyName string, keyVersion string, parameters KeyOperationsParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"vaultBaseUrl": vaultBaseURL,
	}

	pathParameters := map[string]interface{}{
		"key-name":    autorest.Encode("path", keyName),
		"key-version": autorest.Encode("path", keyVersion),
	}

	const APIVersion = "2016-10-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("{vaultBaseUrl}", urlParameters),
		autorest.WithPathParameters("/keys/{key-name}/{key-version}/wrapkey", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// WrapKeySender sends the WrapKey request. The method will close the
// http.Response Body if it receives an error.
func (client ManagementClient) WrapKeySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req)
}

// WrapKeyResponder handles the response to the WrapKey request. The method always
// closes the http.Response Body.
func (client ManagementClient) WrapKeyResponder(resp *http.Response) (result KeyOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}
