#   Convoy is a WSGI app for loading multiple files in the same request.
#   Copyright (C) 2011-2015 Canonical, Ltd.
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU Affero General Public License as
#   published by the Free Software Foundation, either version 3 of the
#   License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU Affero General Public License for more details.
#
#   You should have received a copy of the GNU Affero General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import os.path
import shutil
import tempfile
import unittest

__all__ = [
    "ConvoyTestCase",
]


class ConvoyTestCase(unittest.TestCase):

    def makeDir(self, path=None):
        """Create a temporary directory.

        This directory will be removed when the test completes.

        :param path: An optional directory path to create. If not specified a
            new directory path will be chosen by `tempfile.mkdtemp`.
        :return: The path of the newly created directory.
        """
        if path is None:
            path = tempfile.mkdtemp()
            self.addCleanup(shutil.rmtree, path)
            return path
        else:
            os.makedirs(path)
            return path

    def makeFile(self, content=None, basename=None, dirname=None, path=None):
        """Create a temporary file.

        This file will be removed when the test completes.

        :param content: Optional contents for the new file.
        :param basename: Optional base name for the new file.
        :param dirname: Optional directory in which to put the new file.
        :param path: Optional full path to the new file. Mutually exclusive
            with `basename` and `dirname`.
        :return: The path of the newly created file.
        """
        if path is not None:
            self.assertIsNone(basename)
            self.assertIsNone(dirname)
            self.addCleanup(os.unlink, path)
        elif basename is not None:
            if dirname is None:
                dirname = self.makeDir()
            path = os.path.join(dirname, basename)
        else:
            fileno, path = tempfile.mkstemp(dir=dirname)
            self.addCleanup(os.unlink, path)
            os.close(fileno)

        if content is not None:
            with open(path, "w") as fd:
                fd.write(content)

        return path
