# (c) Copyright 2012-2023, CodeWeavers, Inc.

import multiprocessing
import os
import re
import signal

from gi.repository import Gio
from gi.repository import GLib
from gi.repository import GObject
from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import GdkPixbuf
from gi.repository import Pango

import bottlequery
import bottlewrapper
import c4profilesmanager
import cxdiag
import cxfixes
import cxfsnotifier
import cxlog
import cxmenu
import cxproduct
import cxutils
import demoutils

import applicationsview
import bottlecollection
import bottlemanagement
import bottleoperation
import bottleview
import cxguitools
import cxprefsui
import cxregisterui
import cxrunui
import distversion
import packagebottledialog
import packageview
import ratingdialog
import systeminfo
import updater
import webutils
import pyop

from cxutils import cxgettext as _


class CrossOver(Gtk.Application):
    """This is the CrossOver application"""

    crossover_gui = None
    main_window = None

    update_source = None

    is_demo = False
    updating_checkboxes = False

    _size = None
    _pane_size = None

    selected_bottle = None

    package_view = None
    package_view_controller = None

    bottle_view = None
    bottle_view_controller = None

    applications_view = None
    applications_view_controller = None

    bottles = None
    bottle_list_store = None

    main_menu_model = None
    menubar_model = None
    bottle_context_menu = None
    bottle_context_menu_model = None

    def __init__(self, is_installer=False, is_rating_dialog=False):
        appid = 'com.codeweavers.' + cxproduct.get_cellar_id()
        Gtk.Application.__init__(self, application_id=appid,
                                 flags=Gio.ApplicationFlags.HANDLES_COMMAND_LINE,
                                 register_session=True)

        GLib.set_application_name(distversion.PRODUCT_NAME)

        if is_installer:
            self.add_main_option(
                "bottle",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.STRING,
                "Use the specified bottle and $CX_BOTTLE otherwise",
                "BOTTLE",
            )

            self.add_main_option(
                "tiefile",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.STRING,
                "Load the specified additional .tie file",
                "FILE",
            )

            self.add_main_option(
                "c4pfile",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.STRING,
                "Load the specified additional .tie file",
                "FILE",
            )

            self.add_main_option(
                "profileid",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.STRING,
                "Select the specified profile for installation",
                "PROFILEID",
            )

            self.add_main_option(
                "installersource",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.STRING,
                "Specifies the path to the installer file or directory",
                "FILE",
            )
        elif is_rating_dialog:
            self.add_main_option(
                "bottle",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.STRING,
                "Use the specified bottle and $CX_BOTTLE otherwise",
                "BOTTLE",
            )

            self.add_main_option(
                "reminder",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.NONE,
                "The rating dialog is a reminder",
                None,
            )
        else:
            self.add_main_option(
                "restore",
                0,
                GLib.OptionFlags.NONE,
                GLib.OptionArg.FILENAME,
                "Restore the bottle contained in the specified archive",
                "FILE",
            )
            if distversion.IS_PREVIEW:
                self.add_main_option(
                    "enable-preview",
                    0,
                    GLib.OptionFlags.NONE,
                    GLib.OptionArg.STRING,
                    "Enable preview for the specified bottle",
                    "BOTTLE",
                )

        self.add_main_option(
            "allow-root",
            0,
            GLib.OptionFlags.NONE,
            GLib.OptionArg.NONE,
            "Don't warn about running this tool as root",
            None,
        )

        GLib.unix_signal_add(GLib.PRIORITY_DEFAULT, signal.SIGINT, self.quit)

        self.on_license_changed_observers = {}

    # pylint: disable=W0221
    def do_startup(self):
        Gtk.Application.do_startup(self)

        # Load the main CrossOver window
        self.bottles = bottlecollection.sharedCollection()
        self.bottles.addChangeDelegate(self)
        self.bottles.addBottleChangeDelegate(self)

        self.crossover_gui = Gtk.Builder()
        self.crossover_gui.set_translation_domain("crossover")
        self.crossover_gui.add_from_file(cxguitools.get_ui_path("crossover"))
        self.crossover_gui.connect_signals(self)

        self.initialize_actions()

        self.main_window = self.crossover_gui.get_object("MainWindow")

        global_config = cxproduct.get_config()

        if distversion.IS_CJBUILD:
            first_run_version = global_config["CrossOver"].get("FirstRunVersion")
            if (not first_run_version) or (first_run_version != distversion.CX_VERSION):
                first_run_url = "https://www.crossoverchina.com/redirect/?product=Crossover&version=%s&referer=Start" % distversion.CX_VERSION
                cxutils.launch_url(first_run_url)
                cxproduct.set_config_value('CrossOver', 'FirstRunVersion', distversion.CX_VERSION)

        size = global_config["OfficeSetup"].get("BottleManagerSize")
        if size:
            width = height = None
            if size.count('x') == 1:
                widthstr, heightstr = size.split('x', 1)
                if widthstr.isdigit() and heightstr.isdigit():
                    width = int(widthstr)
                    height = int(heightstr)
                    self.main_window.set_default_size(width, height)
            if width is None:
                cxlog.warn("Invalid BottleManagerSize %s" % cxlog.debug_str(size))

        size = global_config["OfficeSetup"].get("BottleSidebarSize")
        if size and size.isdigit():
            self.crossover_gui.get_object("BottlePaned").set_position(int(size))

        self.crossover_gui.get_object("BottlePaned").connect('notify::position', self.on_bottlepane_move)

        self.package_view_controller = packageview.PackageViewController(self.main_window)
        self.package_view = self.package_view_controller.get_view()
        self.crossover_gui.get_object('ContentView').pack_start(self.package_view, True, True, 0)

        self.bottle_view_controller = bottleview.BottleViewController(self.main_window)
        self.bottle_view = self.bottle_view_controller.get_view()
        self.crossover_gui.get_object('ContentView').pack_start(self.bottle_view, True, True, 0)
        self.bottle_view_controller.get_icon_view().connect('selection-changed', self.on_selection_changed)

        self.applications_view_controller = applicationsview.ApplicationsViewController(self.main_window)
        self.applications_view = self.applications_view_controller.get_view()
        self.crossover_gui.get_object('ContentView').pack_start(self.applications_view, True, True, 0)
        self.applications_view_controller.get_icon_view().connect('selection-changed', self.on_selection_changed)

        self.bottle_list_store = Gio.ListStore.new(BottleListItem)
        self.crossover_gui.get_object('BottleListView').bind_model(self.bottle_list_store, self.create_bottle_list_item)
        self.update_bottle_list()

        self.crossover_gui.get_object('CrossOverIcon').set_from_file(
            os.path.join(cxutils.CX_ROOT, 'share', 'images', 'crossover.png'))

        path = os.path.join(cxutils.CX_ROOT, 'etc', 'license.txt')
        self.on_license_changed_observers[path] = cxfsnotifier.add_observer(path, self.on_license_changed)
        path = os.path.join(cxutils.CX_ROOT, 'etc', 'license.sha256')
        self.on_license_changed_observers[path] = cxfsnotifier.add_observer(path, self.on_license_changed)
        path = os.path.join(cxutils.CX_ROOT, 'etc', 'license.sig')
        self.on_license_changed_observers[path] = cxfsnotifier.add_observer(path, self.on_license_changed)

        self.main_menu_model = Gio.Menu()
        self.crossover_gui.get_object('MenuButton').set_use_popover(False)
        self.crossover_gui.get_object('MenuButton').set_menu_model(self.main_menu_model)
        self.update_main_menu()

        self.bottle_context_menu_model = Gio.Menu()
        self.bottle_context_menu = Gtk.Menu.new_from_model(self.bottle_context_menu_model)
        self.bottle_context_menu.attach_to_widget(self.main_window)
        self.update_bottle_context_menu()

        if global_config['OfficeSetup'].get('MenuBar', '0') == '1':
            self.main_window.set_titlebar(None)
            self.menubar_model = Gio.Menu()
            self.set_menubar(self.menubar_model)
            self.update_menubar()

        self.check_registration()

        self.update_actions()
        self.update_profiles()

        if self.bottles.bottles():
            self.select_all_applications()
        else:
            self.select_install()

    # pylint: disable=W0221
    def do_activate(self):
        # Actually show the main window
        self.add_window(self.main_window)
        self.main_window.present()
        self.set_headerbar()

    def launch_installer(self, command_line, options, bottle_name):
        self.package_view_controller.back()
        self.select_install()

        if bottle_name:
            self.package_view_controller.set_bottle(bottle_name)

        tiefile = None
        if 'tiefile' in options:
            tiefile = options['tiefile']

        if 'c4pfile' in options:
            tiefile = options['c4pfile']

        args = command_line.get_arguments()
        if not tiefile and args and len(args) > 1 and \
           (args[1].endswith('.tie') or args[1].endswith('.gz') or args[1].endswith('c4p')):
            tiefile = args[1]

        if tiefile:
            tiefile = cxutils.abspath(tiefile, command_line.get_cwd())
            self.package_view_controller.parse_c4pfile(tiefile)

        if 'profileid' in options:
            self.package_view_controller.set_profile_from_id(options['profileid'])

        if 'installersource' in options:
            installersource = cxutils.abspath(options['installersource'], command_line.get_cwd())
            self.package_view_controller.set_installer_source(installersource)

    def launch_rating_dialog(self, options, bottle):
        if not bottle or not bottle.appid:
            return

        if 'reminder' in options:
            if not bottle.should_nag_for_ratings():
                return

        controller = ratingdialog.RatingController(bottle, self.main_window, 'reminder' in options)
        self.add_window(controller.get_window())

    def do_command_line(self, command_line):
        options = command_line.get_options_dict()
        # convert GVariantDict -> GVariant -> dict
        options = options.end().unpack()

        if 'allow-root' not in options:
            cxguitools.warn_if_root()

        if 'enable-preview' in options:
            bottle = self.bottles.bottle_with_name(options['enable-preview'])

            if not bottle:
                return 1

            bottleoperation.set_preview_state(bottle, True, self.main_window, wait=True)

            return 0 if bottle.is_preview_enabled() else 1

        if 'restore' in options:
            RestoreArchiveController(self, cxutils.uri_to_path(options['restore']))

        bottle_name = None
        if 'bottle' in options:
            bottle_name = options['bottle']

        if not bottle_name and 'CX_BOTTLE' in os.environ:
            bottle_name = os.environ['CX_BOTTLE']

        args = command_line.get_arguments()
        if args and 'installer' in args[0]:
            self.launch_installer(command_line, options, bottle_name)

        bottle = self.bottles.bottle_with_name(bottle_name)

        if args and 'rating' in args[0]:
            self.launch_rating_dialog(options, bottle)
            return 0

        self.activate()
        return 0

    def present(self):
        self.main_window.present()

    def check_registration(self):
        op = CheckRegistrationOperation(self)
        pyop.sharedOperationQueue.enqueue(op)

    def on_license_changed(self, _event, _path, _observer_data):
        GLib.idle_add(self.check_registration)

    @staticmethod
    def create_bottle_list_item(item):
        return item.get_widget()

    def update_bottle_list(self):
        current_bottles = set(self.bottles.bottles())

        if self.selected_bottle and self.selected_bottle not in current_bottles:
            self.select_all_applications()

        any_published_bottles = any(bottle.is_managed for bottle in current_bottles)
        any_private_bottles = any(not bottle.is_managed for bottle in current_bottles)
        seen_published_section = False
        seen_private_section = False
        reselect_bottle = None

        iterator = 0
        while iterator < self.bottle_list_store.get_n_items():
            item = self.bottle_list_store.get_item(iterator)
            if item.bottle in current_bottles:
                markup = item.get_markup()
                if item.markup != markup:
                    if item.bottle.name == item.bottle_name:
                        # Status change
                        item.set_markup(markup)
                    else:
                        # Bottle rename
                        if self.selected_bottle == item.bottle:
                            reselect_bottle = item.bottle
                        self.bottle_list_store.remove(iterator)
                        continue

                current_bottles.remove(item.bottle)
            elif item.key == BottleListItem.MANAGED_BOTTLES:
                if not any_published_bottles:
                    self.bottle_list_store.remove(iterator)
                    continue

                seen_published_section = True
            elif item.key == BottleListItem.PRIVATE_BOTTLES:
                if not any_private_bottles:
                    self.bottle_list_store.remove(iterator)
                    continue

                seen_private_section = True

                if any_published_bottles:
                    item.set_markup('<b>' + _('Private Bottles') + '</b>')
                else:
                    item.set_markup('<b>' + _('Bottles') + '</b>')
            elif item.bottle:
                self.bottle_list_store.remove(iterator)
                continue

            iterator += 1

        for bottle in current_bottles:
            index = self.bottle_list_store.insert_sorted(BottleListItem(None, bottle),
                                                         BottleListItem.compare)
            if bottle == reselect_bottle:
                self.select_row_at_index(index)

        if any_published_bottles and not seen_published_section:
            markup = '<b>' + _('Published Bottles') + '</b>'
            self.bottle_list_store.insert_sorted(BottleListItem(BottleListItem.MANAGED_BOTTLES, markup=markup),
                                                 BottleListItem.compare)

        if any_private_bottles and not seen_private_section:
            markup = '<b>' + _('Bottles') + '</b>'
            if any_published_bottles:
                markup = '<b>' + _('Private Bottles') + '</b>'

            self.bottle_list_store.insert_sorted(BottleListItem(BottleListItem.PRIVATE_BOTTLES, markup=markup),
                                                 BottleListItem.compare)

    def update_profiles(self):
        config = cxproduct.get_config()
        if config['OfficeSetup'].get('AutoUpdate', '1') != '0':
            op = UpdateCompatDBOperation()
            pyop.sharedOperationQueue.enqueue(op)

            c4p_url = config['OfficeSetup'].get('TieURL', 'http://ftp.codeweavers.com/pub/crossover/tie/crossover.tie.gz')
            if c4p_url:
                op = UpdateProfilesOperation(c4p_url, self)
                pyop.sharedOperationQueue.enqueue(op)
                self.crossover_gui.get_object('UpdatingProfiles').show()

            # Update every day
            if self.update_source is None:
                self.update_source = GLib.timeout_add_seconds(24 * 60 * 60, self.update_profiles)

            return True

        if self.update_source is not None:
            GLib.source_remove(self.update_source)
            self.update_source = None

        return False

    def select_row_at_index(self, index):
        row = self.crossover_gui.get_object('BottleListView').get_row_at_index(index)
        self.crossover_gui.get_object('BottleListView').select_row(row)

        ret = row.translate_coordinates(self.crossover_gui.get_object('BottleListView'), 0, 0)
        if ret and ret[1] >= 0:
            adjustment = self.crossover_gui.get_object('BottleListView').get_adjustment()
            _min, height = row.get_preferred_height()
            adjustment.set_value(ret[1] - (adjustment.get_page_size() - height) / 2)

    def get_selected_launcher(self):
        if self.bottle_view.is_visible():
            return self.bottle_view_controller.selected_launcher
        if self.applications_view.is_visible():
            return self.applications_view_controller.selected_launcher

        return None

    def get_selected_launchers(self):
        if self.bottle_view.is_visible():
            return self.bottle_view_controller.selected_launchers
        if self.applications_view.is_visible():
            return self.applications_view_controller.selected_launchers

        return None

    def get_selected_launcher_is_hidden(self):
        if self.bottle_view.is_visible():
            return self.bottle_view_controller.selected_launcher_is_hidden()
        if self.applications_view.is_visible():
            return self.applications_view_controller.selected_launcher_is_hidden()

        return True

    def update_back_button(self):
        if not self.package_view.get_visible():
            self.crossover_gui.get_object('BackButton').hide()
            return

        selected = self.package_view_controller.installer_selected
        self.crossover_gui.get_object('BackButton').set_visible(selected)

    def on_selection_changed(self, _view):
        self.update_actions()

    def set_menu_from_list(self, menu, menu_list):
        menu.remove_all()
        for section in menu_list:
            section_menu = Gio.Menu()
            for action, item in section.items():
                if isinstance(item, list):
                    submenu = Gio.Menu()
                    self.set_menu_from_list(submenu, item)
                    section_menu.append_submenu(action, submenu)
                elif isinstance(item, Gio.MenuItem):
                    section_menu.append_item(item)
                else:
                    section_menu.append(item, action)

            menu.append_section(None, section_menu)

    def remove_menu_item(self, menu_list, item):
        for submenu in menu_list:
            for key, value in submenu.items():
                if key == item:
                    del submenu[key]
                    return
                if isinstance(value, list):
                    self.remove_menu_item(value, item)

    def create_bottle_menu_list(self):
        if self.selected_bottle:
            name = self.selected_bottle.name.replace('_', '__')

            duplicate_bottle_label = _("_Duplicate '%(bottle)s'…") % {'bottle': name}
            rename_bottle_label = _("Rena_me '%(bottle)s'…") % {'bottle': name}
            delete_bottle_label = _("De_lete '%(bottle)s'…") % {'bottle': name}
            export_bottle_label = _("_Export '%(bottle)s' to Archive…") % {'bottle': name}
            publish_bottle_label = _("Pu_blish '%(bottle)s'…") % {'bottle': name}
            create_package_label = _("Create Pac_kage from '%(bottle)s'…") % {'bottle': name}
            install_into_bottle_label = _("Install _Software into '%(bottle)s'…") % {'bottle': name}
            quit_bottle_label = _("_Quit '%(bottle)s'") % {'bottle': name}
            force_quit_bottle_label = _("_Force Quit '%(bottle)s'") % {'bottle': name}
        else:
            duplicate_bottle_label = _("_Duplicate Bottle…")
            rename_bottle_label = _("Rena_me Bottle…")
            delete_bottle_label = _("De_lete Bottle…")
            export_bottle_label = _("_Export Bottle to Archive…")
            publish_bottle_label = _("Pu_blish Bottle…")
            create_package_label = _("Create Pac_kage from Bottle…")
            install_into_bottle_label = _("Install _Software into Bottle…")
            quit_bottle_label = _("_Quit Bottle")
            force_quit_bottle_label = _("_Force Quit Bottle")

        menu_list = [
            {
                'app.new-bottle': _('_New Bottle…'),
                'app.duplicate-bottle': duplicate_bottle_label,
                'app.rename-bottle': rename_bottle_label,
                'app.delete-bottle': delete_bottle_label,
            },
            {
                'app.open-c-drive': _('Open _C: Drive'),
            },
            {
                'app.export-bottle': export_bottle_label,
                'app.import-bottle': _('_Import Bottle Archive…'),
                'app.publish-bottle': publish_bottle_label,
                'app.update-published-bottle': _('_Update Published Bottle…'),
                'app.create-package': create_package_label,
            },
            {
                _('Se_ttings'): [
                    {
                        'app.default-bottle': _('_Default Bottle'),
                    },
                    {
                        'app.graphics-auto': _("_Automatic graphics backend"),
                        'app.graphics-dxvk': _("DX_VK graphics backend"),
                        'app.graphics-wine': _("_Wine graphics backend"),
                    },
                    {
                        'app.toggle-esync': _('Performance Enhanced Synchronization (_ESync)'),
                        'app.toggle-highres': _('_High Resolution Mode'),
                        'app.toggle-preview': _('CrossOver Preview'),
                    },
                    {
                        'app.reset-browser': _('_Reset Default Web Browser to Linux Browser'),
                    }
                ]
            },
            {
                'app.install-into-bottle': install_into_bottle_label,
            },
            {
                'app.run-command': _('_Run Command…'),
            },
            {
                'app.quit-bottle': quit_bottle_label,
                'app.force-quit-bottle': force_quit_bottle_label,
            }
        ]

        global_config = cxproduct.get_config()
        if not distversion.HAS_MULTI_USER or (
                not cxproduct.is_root_install() and
                global_config["OfficeSetup"].get("NonRootPublish", "0") != "1"):
            self.remove_menu_item(menu_list, 'app.publish-bottle')
            self.remove_menu_item(menu_list, 'app.update-published-bottle')

        if not distversion.IS_PREVIEW:
            self.remove_menu_item(menu_list, 'app.toggle-preview')

        return menu_list

    def create_file_menu_list(self):
        launcher = self.get_selected_launcher()

        name = None
        if isinstance(launcher, cxmenu.MenuItem):
            name = launcher.menu_name().replace('_', '__')

        open_label = _("_Open")
        if isinstance(launcher, cxmenu.MenuItem):
            open_label = _("_Open '%s'") % name

        run_with_options_label = _("_Run with Options…")
        if isinstance(launcher, cxmenu.MenuItem) and launcher.type == 'windows':
            run_with_options_label = _("_Run '%s' with Options…") % name

        toggle_hidden_label = _("_Hide from 'Home'")
        if self.get_selected_launchers():
            if self.get_selected_launcher_is_hidden():
                toggle_hidden_label = _("_Show '%s' in 'Home'") % name
            else:
                toggle_hidden_label = _("_Hide '%s' from 'Home'") % name

        return [
            {
                'app.open': open_label,
                'app.run-with-options': run_with_options_label,
            },
            {
                'app.toggle-hidden': toggle_hidden_label,
            }
        ]

    def update_main_menu(self):
        menu_list = [
            {
                _('_File'): self.create_file_menu_list(),
                _('_Bottle'): self.create_bottle_menu_list(),
                _('_Help'): [
                    {
                        'app.help': _('CrossOver _Help'),
                        'app.system-information': _('_System Information'),
                        'app.about': _('_About'),
                    }
                ],
                'app.register': _('_Unlock CrossOver…'),
                'app.update': _('_Check for Updates…'),
                'app.preferences': _('_Preferences'),
                'app.quit': _('_Quit'),
            }
        ]

        if self.is_demo:
            self.remove_menu_item(menu_list, 'app.update')
        else:
            self.remove_menu_item(menu_list, 'app.register')

        if distversion.IS_CJBUILD:
            self.remove_menu_item(menu_list, 'app.update')

        self.set_menu_from_list(self.main_menu_model, menu_list)

    def update_menubar(self):
        file_menu_list = self.create_file_menu_list()
        file_menu_list.append({
            'app.quit': _('_Quit')
        })

        menu_list = [
            {
                _('_File'): file_menu_list,
                _('_Edit'): [
                    {
                        'app.preferences': _('_Preferences'),
                    }
                ],
                _('_Bottle'): self.create_bottle_menu_list(),
                _('_Help'): [
                    {
                        'app.help': _('CrossOver _Help'),
                        'app.register': _('_Unlock CrossOver…'),
                        'app.update': _('_Check for Updates…'),
                        'app.system-information': _('_System Information'),
                        'app.about': _('_About'),
                    }
                ],
            }
        ]

        if self.is_demo:
            self.remove_menu_item(menu_list, 'app.update')
        else:
            self.remove_menu_item(menu_list, 'app.register')

        if distversion.IS_CJBUILD:
            self.remove_menu_item(menu_list, 'app.update')

        self.set_menu_from_list(self.menubar_model, menu_list)

    def update_bottle_context_menu(self):
        menu_list = self.create_bottle_menu_list()

        self.remove_menu_item(menu_list, 'app.new-bottle')
        self.remove_menu_item(menu_list, 'app.import-bottle')

        self.set_menu_from_list(self.bottle_context_menu_model, menu_list)

    def select_all_applications(self):
        self.crossover_gui.get_object('HeaderListView').select_row(self.crossover_gui.get_object('AllApplicationsButton'))

    def select_install(self):
        self.crossover_gui.get_object('HeaderListView').select_row(self.crossover_gui.get_object('InstallButton'))

    def set_headerbar(self):
        self.crossover_gui.get_object('HeaderBar').set_title(distversion.PRODUCT_NAME)

        pos = self.crossover_gui.get_object('HeaderBarSeparator').translate_coordinates(
            self.crossover_gui.get_object('HeaderBarSpacing'), 0, 0)
        if not pos:
            return

        distance = pos[0] - self.crossover_gui.get_object('HeaderBarSpacing').get_allocated_width()
        pane_size = self.crossover_gui.get_object('BottlePaned').get_position()
        x, _y = self.crossover_gui.get_object('HeaderBar').translate_coordinates(
            self.crossover_gui.get_object('HeaderBarSpacing'), pane_size, 0)
        x = max(-1, x - distance)

        self.crossover_gui.get_object('HeaderBarSpacing').set_size_request(x, -1)

    def bottleCollectionChanged(self):
        self.update_bottle_list()
        self.update_actions()

    def bottleChanged(self, bottle):
        # This can be called from outside the main thread
        GLib.idle_add(self.update_bottle_list)
        if bottle == self.selected_bottle:
            GLib.idle_add(self.update_actions)

    def reveal_bottle(self, bottle):
        for i in range(self.bottle_list_store.get_n_items()):
            item = self.bottle_list_store.get_item(i)
            if item.bottle == bottle:
                self.select_row_at_index(i)

        self.package_view_controller.back()

    def on_MenuButton_toggled(self, button):
        if button.get_active():
            self.update_main_menu()

    def on_MainWindow_configure_event(self, _widget, _event):
        # get_size doesn't work in on_destroy
        self._size = self.main_window.get_size()

    def on_bottlepane_move(self, _obj, _propertyspec):
        self._pane_size = self.crossover_gui.get_object("BottlePaned").get_position()
        self.set_headerbar()

    def on_destroy(self, _unused):
        for path, observer_id in self.on_license_changed_observers.items():
            cxfsnotifier.remove_observer(path, observer_id)

        self.bottles.removeChangeDelegate(self)
        if self._size:
            cxproduct.set_config_value('OfficeSetup', 'BottleManagerSize', '%sx%s' % self._size)
        if self._pane_size:
            cxproduct.set_config_value('OfficeSetup', 'BottleSidebarSize', str(self._pane_size))

    def on_BottleListView_button_press_event(self, view, event):
        row = view.get_row_at_y(int(event.y))
        if row is None:
            return False # propagate

        item = self.bottle_list_store.get_item(row.get_index())

        if event.button == 1: # left click
            if item and item.bottle and item.bottle == self.selected_bottle:
                item.edit()
                return True

        if event.button == 3: # right click
            view.select_row(row)

            if not item or not item.bottle:
                return True

            self.update_bottle_context_menu()
            cxguitools.popup_at_pointer(self.bottle_context_menu, view, event)
            return True

        return False # propagate

    def on_BottleListView_popup_menu(self, widget):
        self.update_bottle_context_menu()

        cxguitools.popup_at_widget(
            self.bottle_context_menu,
            widget.get_selected_row(),
            Gdk.Gravity.CENTER,
            Gdk.Gravity.NORTH_WEST,
            None)

        return True

    def on_BottleListView_row_selected(self, _widget, child):
        if not child:
            return

        item = self.bottle_list_store.get_item(child.get_index())
        if self.selected_bottle is not item.bottle:
            self.crossover_gui.get_object('HeaderListView').unselect_all()
            self.selected_bottle = item.bottle

            if self.bottle_view_controller.bottle != item.bottle:
                self.bottle_view_controller.set_bottle(item.bottle)

            self.applications_view.hide()
            self.package_view.hide()
            self.bottle_view.show()

            if not self.package_view_controller.installer_selected:
                self.package_view_controller.back()

            self.update_back_button()
            self.update_actions()

    def on_HeaderListView_row_selected(self, _widget, child):
        if not child:
            return

        self.crossover_gui.get_object('BottleListView').unselect_all()
        self.selected_bottle = None

        if child == self.crossover_gui.get_object('AllApplicationsButton'):
            self.applications_view.show()
            self.package_view.hide()

            if not self.package_view_controller.installer_selected:
                self.package_view_controller.back()
        else:
            self.applications_view.hide()
            self.package_view.show()

        self.bottle_view.hide()

        self.update_back_button()
        self.update_actions()

    # Initialize actions

    def initialize_actions(self):
        action_entries = {
            'back': self.on_back_activate,
            'help': self.on_help_activate,
            'system-information': self.on_system_information_activate,
            'about': self.on_about_activate,
            'register': self.on_register_activate,
            'update': self.on_update_activate,
            'preferences': self.on_preferences_activate,
            'quit': self.on_quit_activate,
            'open': self.on_open_activate,
            'run-command': self.on_run_command_activate,
            'run-with-options': self.on_run_with_options_activate,
            'toggle-hidden': self.on_toggle_hidden_activate,
            'new-bottle': self.on_new_bottle_activate,
            'duplicate-bottle': self.on_duplicate_bottle_activate,
            'rename-bottle': self.on_rename_bottle_activate,
            'delete-bottle': self.on_delete_bottle_activate,
            'open-c-drive': self.on_open_c_drive_activate,
            'export-bottle': self.on_export_bottle_activate,
            'import-bottle': self.on_import_bottle_activate,
            'publish-bottle': self.on_publish_bottle_activate,
            'update-published-bottle': self.on_update_published_bottle_activate,
            'create-package': self.on_create_package_activate,
            'reset-browser': self.on_reset_browser_activate,
            'install-into-bottle': self.on_install_activate,
            'install-missing': self.on_install_missing_activate,
            'quit-bottle': self.on_quit_bottle_activate,
            'force-quit-bottle': self.on_force_quit_bottle_activate,
        }

        for name, callback in action_entries.items():
            action = Gio.SimpleAction.new(name, None)
            action.connect('activate', callback)
            self.add_action(action)

        toggle_entries = {
            'default-bottle': self.on_default_bottle_change_state,
            'graphics-auto': self.on_graphics_auto_change_state,
            'graphics-dxvk': self.on_graphics_dxvk_change_state,
            'graphics-wine': self.on_graphics_wine_change_state,
            'toggle-esync': self.on_esync_change_state,
            'toggle-highres': self.on_highres_change_state,
            'toggle-preview': self.on_preview_change_state,
            'toggle-auto-update': self.on_auto_update_change_state,
            'toggle-update-check': self.on_update_check_change_state,
            'toggle-ratings': self.on_ratings_change_state,
            'toggle-show-untested': self.on_show_untested_change_state,
            'toggle-cxfixes': self.on_cxfixes_change_state,
            'toggle-menubar': self.on_menubar_change_state,
        }

        for name, callback in toggle_entries.items():
            action = Gio.SimpleAction.new_stateful(name, None, GLib.Variant.new_boolean(False))
            action.connect('change-state', callback)
            self.add_action(action)

        action = Gio.SimpleAction.new_stateful('graphics', None, GLib.Variant.new_string('auto'))
        action.connect('change-state', self.on_graphics_change_state)
        self.add_action(action)

        action = Gio.SimpleAction.new('open-file', GLib.VariantType.new('s'))
        action.connect('activate', self.on_open_file_activate)
        self.add_action(action)

        global_config = cxproduct.get_config()
        self.lookup_action('toggle-auto-update').set_state(GLib.Variant.new_boolean(
            global_config['OfficeSetup'].get('AutoUpdate', '1') == '1'))
        self.lookup_action('toggle-update-check').set_state(GLib.Variant.new_boolean(
            global_config['CrossOver'].get('CheckForUpdates', '1') == '1'))
        self.lookup_action('toggle-ratings').set_state(GLib.Variant.new_boolean(
            global_config['CrossOver'].get('AskForRatings', '1') == '1'))
        self.lookup_action('toggle-show-untested').set_state(GLib.Variant.new_boolean(
            global_config['OfficeSetup'].get('ShowUntestedApps', '1') == '1'))
        self.lookup_action('toggle-cxfixes').set_state(GLib.Variant.new_boolean(
            global_config['OfficeSetup'].get('ApplyCxfixes', '1') == '1'))
        self.lookup_action('toggle-menubar').set_state(GLib.Variant.new_boolean(
            global_config['OfficeSetup'].get('MenuBar', '0') == '1'))

    # Update actions

    def update_graphics_toggle(self):
        state = bottlewrapper.BottleWrapper.STATUS_GRAPHICS_BACKEND_UNKNOWN
        if self.selected_bottle:
            state = self.selected_bottle.graphics_backend_state

        if state == bottlewrapper.BottleWrapper.STATUS_GRAPHICS_BACKEND_DXVK:
            self.lookup_action('graphics').set_state(GLib.Variant.new_string('dxvk'))
            self.lookup_action('graphics-auto').set_state(GLib.Variant.new_boolean(False))
            self.lookup_action('graphics-dxvk').set_state(GLib.Variant.new_boolean(True))
            self.lookup_action('graphics-wine').set_state(GLib.Variant.new_boolean(False))
        elif state == bottlewrapper.BottleWrapper.STATUS_GRAPHICS_BACKEND_WINE:
            self.lookup_action('graphics').set_state(GLib.Variant.new_string('wine'))
            self.lookup_action('graphics-auto').set_state(GLib.Variant.new_boolean(False))
            self.lookup_action('graphics-dxvk').set_state(GLib.Variant.new_boolean(False))
            self.lookup_action('graphics-wine').set_state(GLib.Variant.new_boolean(True))
        else:
            self.lookup_action('graphics').set_state(GLib.Variant.new_string('auto'))
            self.lookup_action('graphics-auto').set_state(GLib.Variant.new_boolean(True))
            self.lookup_action('graphics-dxvk').set_state(GLib.Variant.new_boolean(False))
            self.lookup_action('graphics-wine').set_state(GLib.Variant.new_boolean(False))

    def update_esync_toggle(self):
        state = bottlewrapper.BottleWrapper.STATUS_ESYNC_UNKNOWN
        if self.selected_bottle:
            state = self.selected_bottle.is_esync_enabled_state

        if state == bottlewrapper.BottleWrapper.STATUS_ESYNC_UNKNOWN:
            self.lookup_action('toggle-esync').set_enabled(False)
            self.lookup_action('toggle-esync').set_state(GLib.Variant.new_boolean(False))
        elif state == bottlewrapper.BottleWrapper.STATUS_ESYNC_ENABLED:
            self.lookup_action('toggle-esync').set_enabled(True)
            self.lookup_action('toggle-esync').set_state(GLib.Variant.new_boolean(True))
        elif state == bottlewrapper.BottleWrapper.STATUS_ESYNC_DISABLED:
            self.lookup_action('toggle-esync').set_enabled(True)
            self.lookup_action('toggle-esync').set_state(GLib.Variant.new_boolean(False))

    def update_highres_toggle(self):
        state = bottlewrapper.BottleWrapper.STATUS_HIGHRES_UNKNOWN
        if self.selected_bottle:
            state = self.selected_bottle.is_high_resolution_enabled_state

        if state in (bottlewrapper.BottleWrapper.STATUS_HIGHRES_UNKNOWN,
                     bottlewrapper.BottleWrapper.STATUS_HIGHRES_UNAVAILABLE):
            self.lookup_action('toggle-highres').set_enabled(False)
            self.lookup_action('toggle-highres').set_state(GLib.Variant.new_boolean(False))
        elif state == bottlewrapper.BottleWrapper.STATUS_HIGHRES_ENABLED:
            self.lookup_action('toggle-highres').set_enabled(True)
            self.lookup_action('toggle-highres').set_state(GLib.Variant.new_boolean(True))
        elif state == bottlewrapper.BottleWrapper.STATUS_HIGHRES_DISABLED:
            self.lookup_action('toggle-highres').set_enabled(True)
            self.lookup_action('toggle-highres').set_state(GLib.Variant.new_boolean(False))

    def update_preview_toggle(self):
        state = bottlewrapper.BottleWrapper.STATUS_PREVIEW_UNKNOWN
        if self.selected_bottle:
            state = self.selected_bottle.is_preview_enabled_state

        if state == bottlewrapper.BottleWrapper.STATUS_PREVIEW_UNKNOWN:
            self.lookup_action('toggle-preview').set_enabled(False)
            self.lookup_action('toggle-preview').set_state(GLib.Variant.new_boolean(False))
        elif state == bottlewrapper.BottleWrapper.STATUS_PREVIEW_ENABLED:
            self.lookup_action('toggle-preview').set_enabled(True)
            self.lookup_action('toggle-preview').set_state(GLib.Variant.new_boolean(True))
        elif state == bottlewrapper.BottleWrapper.STATUS_PREVIEW_DISABLED:
            self.lookup_action('toggle-preview').set_enabled(True)
            self.lookup_action('toggle-preview').set_state(GLib.Variant.new_boolean(False))

    def update_actions(self):
        self.updating_checkboxes = True

        launcher = self.get_selected_launcher()

        cond = isinstance(launcher, cxmenu.MenuItem)
        self.lookup_action('open').set_enabled(cond)

        cond = isinstance(launcher, cxmenu.MenuItem) and launcher.type == 'windows' and \
            (not self.selected_bottle or self.selected_bottle.can_run_commands)
        self.lookup_action('run-with-options').set_enabled(cond)

        cond = bool(self.get_selected_launchers())
        self.lookup_action('toggle-hidden').set_enabled(cond)

        cond = not self.selected_bottle or self.selected_bottle.can_run_commands
        self.lookup_action('run-command').set_enabled(cond)

        # Update the 'default bottle' checkbox to match the selected bottle
        cond = self.selected_bottle is not None and self.selected_bottle.is_default
        self.lookup_action('default-bottle').set_state(GLib.Variant.new_boolean(cond))

        cond = self.selected_bottle is not None and not self.selected_bottle.is_busy
        for item in ('duplicate-bottle', 'delete-bottle',
                     'open-c-drive', 'export-bottle',
                     'create-package', 'default-bottle'):
            self.lookup_action(item).set_enabled(cond)

        cond = self.selected_bottle is not None and \
            not self.selected_bottle.is_managed and \
            self.selected_bottle.can_run_commands
        self.lookup_action('reset-browser').set_enabled(cond)

        cond = self.selected_bottle is not None and self.selected_bottle.is_active
        self.lookup_action('quit-bottle').set_enabled(cond)

        # Only allow force-quitting if a regular quit has been attempted
        cond = self.selected_bottle is not None and self.selected_bottle.can_force_quit
        self.lookup_action('force-quit-bottle').set_enabled(cond)

        self.update_graphics_toggle()
        self.update_esync_toggle()
        self.update_highres_toggle()
        self.update_preview_toggle()

        self.updating_checkboxes = False

        # Only allow updating if the current bottle is managed and out-of-date
        cond = self.selected_bottle is not None and \
            self.selected_bottle.is_managed and \
            not self.selected_bottle.up_to_date and \
            self.selected_bottle.is_usable

        self.lookup_action('update-published-bottle').set_enabled(cond)

        cond = self.selected_bottle is not None and self.selected_bottle.can_rename
        self.lookup_action('rename-bottle').set_enabled(cond)

        cond = self.selected_bottle is not None and \
            self.selected_bottle.can_install

        self.lookup_action('install-into-bottle').set_enabled(cond)
        self.lookup_action('install-missing').set_enabled(cond)

        # Disable the operations that cannot be performed on managed bottles
        cond = self.selected_bottle is not None and \
            not self.selected_bottle.is_managed and \
            self.selected_bottle.is_usable

        self.lookup_action('publish-bottle').set_enabled(cond)

        cond = self.selected_bottle is not None and \
            not self.selected_bottle.is_managed and \
            not self.selected_bottle.is_disabled

        for item in ('graphics-auto', 'graphics-dxvk', 'toggle-esync', 'toggle-highres'):
            self.lookup_action(item).set_enabled(cond)

        cond = self.selected_bottle is not None and \
            not self.selected_bottle.is_managed

        self.lookup_action('toggle-preview').set_enabled(cond)

        if self.selected_bottle:
            self.selected_bottle.load_high_resolution_info()

        if self.menubar_model:
            self.update_menubar()

    # Action callbacks

    def on_back_activate(self, _action, _parameter):
        self.package_view_controller.back()

    @staticmethod
    def on_help_activate(_action, _parameter):
        cxguitools.show_help()

    @staticmethod
    def on_system_information_activate(_action, _parameter):
        SystemInfoDialog()

    def on_about_activate(self, _action, _parameter):
        dialog = Gtk.AboutDialog(title=_('About CrossOver'),
                                 transient_for=self.main_window,
                                 destroy_with_parent=True)

        logo = GdkPixbuf.Pixbuf.new_from_file(os.path.join(cxutils.CX_ROOT,
                                                           'share',
                                                           'images',
                                                           'welcomeCrossOverIcon.png'))
        dialog.set_logo(logo)

        dialog.set_copyright('© ' + distversion.PRODUCT_COPYRIGHT + '\n\n\n\n\n\n\n\n\n\n\n\n')
        dialog.set_version(distversion.PUBLIC_VERSION)

        dialog.set_website('https://www.codeweavers.com/privacy-policy')
        dialog.set_website_label(_('Privacy Policy'))

        try:
            with open(os.path.join(cxutils.CX_ROOT, 'license.txt'), 'r', encoding='utf8') as f:
                license_text = f.read()
        except OSError:
            # debian
            import gzip
            with gzip.open(os.path.join(cxutils.CX_ROOT, 'doc', 'license.txt.gz'), 'r') as f:
                license_text = cxutils.string_to_str(f.read())

        license_text = re.sub(r'[(](?P<url>http[^) ]+)[)]', r'<\g<url>>', license_text)
        license_text = re.sub(r' (?P<url>http\S+)(?P<eos>\.\s)', r'<\g<url>>\g<eos>', license_text)
        dialog.set_license(license_text)

        dialog.run()

        dialog.destroy()

    @staticmethod
    def on_register_activate(_action, _parameter):
        cxregisterui.open_or_show()

    def on_update_activate(self, _action, _parameter):
        updater.UpdateController(self.main_window)

    def on_preferences_activate(self, _action, _parameter):
        dialog = cxprefsui.PreferencesDialogController(self.main_window)
        self.add_window(dialog.get_window())

    def on_quit_activate(self, _action, _parameter):
        self.main_window.destroy()

    def on_open_activate(self, _action, _parameter):
        launcher = self.get_selected_launcher()
        if launcher:
            launcher.start()

    def on_run_command_activate(self, _action, parameter):
        if self.selected_bottle:
            controller = cxrunui.RunCommandController(self.selected_bottle.name)
        else:
            controller = cxrunui.RunCommandController(None)

        if parameter:
            controller.set_command(parameter.get_string())

    def on_run_with_options_activate(self, _action, _parameter):
        launcher = self.get_selected_launcher()
        lnkfile = launcher.lnkfile
        if not lnkfile:
            return

        bottle = launcher.parent.bottlename
        cxrunui.RunCommandController(bottle).set_launcher(launcher)

    def on_toggle_hidden_activate(self, _action, _parameter):
        if self.bottle_view.is_visible():
            self.bottle_view_controller.toggle_hidden()
        elif self.applications_view.is_visible():
            self.applications_view_controller.toggle_hidden()

    def on_new_bottle_activate(self, _action, _parameter):
        NewBottleController(self)

    def on_duplicate_bottle_activate(self, _action, _parameter):
        if self.selected_bottle:
            NewBottleController(self, bottle_to_copy=self.selected_bottle.name)

    def on_rename_bottle_activate(self, _action, _parameter):
        if not self.selected_bottle or not self.selected_bottle.is_usable:
            return

        row = self.crossover_gui.get_object('BottleListView').get_selected_row()
        if not row:
            return

        item = self.bottle_list_store.get_item(row.get_index())
        item.edit()

    def on_delete_bottle_activate(self, _action, _parameter):
        bottleoperation.delete_bottle(self.selected_bottle, self.main_window)

    def on_open_c_drive_activate(self, _action, _parameter):
        if not self.selected_bottle:
            return

        bottleoperation.open_c_drive(self.selected_bottle)

    def on_export_bottle_activate(self, _action, _parameter):
        if not self.selected_bottle or not self.selected_bottle.is_usable:
            return

        bottleoperation.archive_bottle(self.selected_bottle, self.main_window)

    def on_import_bottle_activate(self, _action, _parameter):
        RestoreArchiveController(self)

    def on_publish_bottle_activate(self, _action, _parameter):
        if not self.selected_bottle or not self.selected_bottle.is_usable:
            return

        PublishBottleController(self, self.selected_bottle.name)

    def on_update_published_bottle_activate(self, _action, _parameter):
        if not self.selected_bottle or not self.selected_bottle.is_usable or self.selected_bottle.up_to_date:
            return

        pyop.sharedOperationQueue.enqueue(UpgradeBottleOperation(self.selected_bottle))

    def on_create_package_activate(self, _action, _parameter):
        if not self.selected_bottle or not self.selected_bottle.is_usable:
            return

        if ' ' in self.selected_bottle.name:
            cxguitools.CXMessageDlg(
                _("The bottle '%s' cannot be packaged because it has a space in the name. Please rename it and try again.") % self.selected_bottle.name,
                buttons=Gtk.ButtonsType.OK,
                parent=self.main_window,
                message_type=Gtk.MessageType.ERROR)
            return

        packagebottledialog.PackageBottleController(self.selected_bottle, self.main_window)

    def on_reset_browser_activate(self, _action, _parameter):
        if not self.selected_bottle:
            return

        bottlequery.set_native_browser(self.selected_bottle.name)

    def on_install_activate(self, _action, parameter):
        if self.selected_bottle and (not self.selected_bottle.is_usable or self.selected_bottle.is_managed):
            return

        self.package_view_controller.back()

        if self.selected_bottle:
            self.package_view_controller.set_bottle(self.selected_bottle.name)

        if parameter:
            self.package_view_controller.parse_c4pfile(parameter.get_string())

        self.select_install()

    def on_install_missing_activate(self, _action, _parameter):
        if self.selected_bottle and (not self.selected_bottle.is_usable or self.selected_bottle.is_managed):
            return

        self.package_view_controller.back()

        if self.selected_bottle:
            profile = self.selected_bottle.get_missing_dependencies_profile()
            if not profile:
                return

            self.package_view_controller.set_bottle(self.selected_bottle.name)
            self.package_view_controller.set_profile(profile)

        self.select_install()

    def quit_bottle_callback(self):
        self.update_bottle_list()
        self.update_actions()

    def on_quit_bottle_activate(self, _action, _parameter):
        if not self.selected_bottle or not self.selected_bottle.is_usable:
            return

        bottleoperation.quit_bottle(self.selected_bottle, False, self.quit_bottle_callback)

    def on_force_quit_bottle_activate(self, _action, _parameter):
        if not self.selected_bottle:
            return

        bottleoperation.quit_bottle(self.selected_bottle, True, self.quit_bottle_callback)

    def on_open_file_activate(self, action, parameter):
        filename = parameter.get_string()
        if cxguitools.filter_path('tie', filename):
            self.on_install_activate(action, parameter)
            return

        profiles = c4profilesmanager.get_matching_profiles(filename)
        if len(profiles) == 1:
            self.package_view_controller.set_profile_from_id(next(iter(profiles.keys())))
            self.package_view_controller.set_installer_source(filename)
            self.select_install()
            return

        self.on_run_command_activate(action, parameter)

    def on_default_bottle_change_state(self, action, value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        action.set_state(value)
        state = value.get_boolean()

        if state != self.selected_bottle.is_default:
            self.selected_bottle.set_is_default(state, self.bottles.refreshDefaultBottle)

    def on_graphics_change_state(self, action, value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        if value == action.get_state():
            return

        action.set_state(value)
        state = value.get_string()

        if state == 'dxvk':
            self.selected_bottle.set_graphics_backend_dxvk()
        elif state == 'wine':
            self.selected_bottle.set_graphics_backend_wine()
        else:
            self.selected_bottle.set_graphics_backend_auto()

        self.update_graphics_toggle()

    def on_graphics_auto_change_state(self, _action, _value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        self.lookup_action('graphics').change_state(GLib.Variant.new_string('auto'))

    def on_graphics_dxvk_change_state(self, _action, _value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        self.lookup_action('graphics').change_state(GLib.Variant.new_string('dxvk'))

    def on_graphics_wine_change_state(self, _action, _value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        self.lookup_action('graphics').change_state(GLib.Variant.new_string('wine'))

    def on_esync_change_state(self, action, value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        action.set_state(value)
        state = value.get_boolean()

        bottleoperation.set_esync_state(self.selected_bottle, state, self.main_window)

    def on_highres_change_state(self, action, value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        action.set_state(value)
        state = value.get_boolean()

        bottleoperation.set_highres_state(self.selected_bottle, state, self.main_window)

    def on_preview_change_state(self, action, value):
        if self.updating_checkboxes or not self.selected_bottle:
            return

        action.set_state(value)
        state = value.get_boolean()

        bottleoperation.set_preview_state(self.selected_bottle, state, self.main_window)

    def on_auto_update_change_state(self, action, value):
        action.set_state(value)
        cxproduct.set_config_value('OfficeSetup', 'AutoUpdate', '1' if value.get_boolean() else '0')

        self.update_profiles()

    @staticmethod
    def on_update_check_change_state(action, value):
        action.set_state(value)
        cxproduct.set_config_value('CrossOver', 'CheckForUpdates', '1' if value.get_boolean() else '0')

    @staticmethod
    def on_ratings_change_state(action, value):
        action.set_state(value)
        cxproduct.set_config_value('CrossOver', 'AskForRatings', '1' if value.get_boolean() else '0')

    def on_show_untested_change_state(self, action, value):
        action.set_state(value)
        cxproduct.set_config_value('OfficeSetup', 'ShowUntestedApps', '1' if value.get_boolean() else '0')

        self.package_view_controller.update_profiles()

    @staticmethod
    def on_cxfixes_change_state(action, value):
        action.set_state(value)
        cxproduct.set_config_value('OfficeSetup', 'ApplyCxfixes', '1' if value.get_boolean() else '0')

    @staticmethod
    def on_menubar_change_state(action, value):
        action.set_state(value)
        cxproduct.set_config_value('OfficeSetup', 'MenuBar', '1' if value.get_boolean() else '0')


def run_cxfixes(check_64bit=True):
    global_config = cxproduct.get_config()
    if global_config['OfficeSetup'].get('ApplyCxfixes', '1') == '1':
        flags = cxdiag.CHECK_32BIT
        if check_64bit:
            flags |= cxdiag.CHECK_64BIT
        diag = cxdiag.get(None, flags)
        cxfixes.clear_errors()
        for errid, title in diag.warnings.items():
            lib = diag.libs.get(errid, None)
            level = cxfixes.get_error_level(errid)
            if level in ('required', 'recommended'):
                cxfixes.add_error(errid, title, level, lib)
        cxfixes.add_masked_errors(('required', 'recommended'))
        if cxfixes.fix_errors() is not None:
            cxdiag.clear()


class NewBottleController:

    def __init__(self, main_window, bottle_to_copy=None):
        self.main_window = main_window

        self.timeout_source = None

        self.modaldialogxml = Gtk.Builder()
        self.modaldialogxml.set_translation_domain("crossover")
        self.modaldialogxml.add_from_file(cxguitools.get_ui_path("newbottledialog"))
        self.modaldialogxml.connect_signals(self)
        newbottledlg = self.modaldialogxml.get_object("NewBottleDialog")

        newbottledlg.props.transient_for = main_window.main_window
        newbottledlg.connect('destroy', self.on_destroy)

        self.bottle_to_copy = bottle_to_copy
        templatewidget = self.modaldialogxml.get_object("bottletype")
        templatelabel = self.modaldialogxml.get_object("bottletypelabel")
        caption = self.modaldialogxml.get_object("newbottlecaption")
        bottleNameWidget = self.modaldialogxml.get_object("newbottlename")
        if bottle_to_copy:
            newbottledlg.set_title(_("Copy a Bottle"))
            caption.set_text(_("This will copy the '%(bottle)s' bottle. The copy will be named:") % {'bottle': bottle_to_copy})
            templatewidget.hide()
            templatelabel.hide()
            self.modaldialogxml.get_object("bottlenamelabel").hide()

            copyBottleName = _("Copy of %(bottle)s") % {'bottle': bottle_to_copy}
            # Eliminate spaces from the bottle being copied while we're at it
            copyBottleName = copyBottleName.replace(' ', '_')
            bottleNameWidget.set_text(copyBottleName)

        else:
            caption.hide()

            liststore = Gtk.ListStore(str, str) # display name, template
            templatewidget.set_model(liststore)
            index = 0
            for template in sorted(bottlemanagement.template_list(),
                                   key=bottlemanagement.get_template_key):
                display_name = bottlemanagement.get_template_name(template)
                new_row = liststore.append()
                liststore.set_value(new_row, 0, display_name)
                liststore.set_value(new_row, 1, template)
                if template == "win10_64":
                    templatewidget.set_active(index)
                index += 1
            cell = Gtk.CellRendererText()
            templatewidget.pack_start(cell, True)
            templatewidget.add_attribute(cell, "text", 0)

            bottle_name_hint = _("New Bottle")
            bottle_name_hint = bottle_name_hint.replace(' ', '_')
            bottleNameWidget.set_text(bottlequery.unique_bottle_name(bottle_name_hint))

        progbar = self.modaldialogxml.get_object("NewBottleProgbar")
        progbar.hide()

        newbottledlg.show()

    def bottle_name_delete_text(self, caller, start, stop):
        name = caller.get_text()
        name = name[:start] + name[stop:]
        if not cxutils.is_valid_bottlename(name):
            caller.stop_emission_by_name("delete-text")
        else:
            okButton = self.modaldialogxml.get_object("okbutton1")
            okButton.set_sensitive(bottlequery.is_valid_new_bottle_name(name))

    def bottle_name_insert_text(self, caller, new_text, _length, _user_data):
        name = caller.get_text()
        position = caller.get_position()
        name = name[:position] + new_text + name[position:]
        if not cxutils.is_valid_bottlename(name):
            caller.stop_emission_by_name("insert-text")
        else:
            okButton = self.modaldialogxml.get_object("okbutton1")
            okButton.set_sensitive(bottlequery.is_valid_new_bottle_name(name))

    def add_bottle_ok(self, _caller):
        bottleNameWidget = self.modaldialogxml.get_object("newbottlename")
        progbar = self.modaldialogxml.get_object("NewBottleProgbar")
        cancelButton = self.modaldialogxml.get_object("cancelbutton1")
        okButton = self.modaldialogxml.get_object("okbutton1")
        templatewidget = self.modaldialogxml.get_object("bottletype")

        if self.bottle_to_copy is None:
            templatemodel = templatewidget.get_model()
            template = templatemodel.get_value(templatewidget.get_active_iter(), 1)
        else:
            template = None

        newBottleOp = NewBottleOperation(cxutils.string_to_unicode(bottleNameWidget.get_text()), template, self, self.bottle_to_copy)
        progbar.show()
        bottleNameWidget.set_sensitive(False)
        templatewidget.set_sensitive(False)
        cancelButton.set_sensitive(False)
        okButton.set_sensitive(False)

        self.timeout_source = GLib.timeout_add(100, self.pulse)

        pyop.sharedOperationQueue.enqueue(newBottleOp)

    def pulse(self):
        self.modaldialogxml.get_object("NewBottleProgbar").pulse()
        return True

    def add_bottle_cancel(self, _caller):
        newbottledlg = self.modaldialogxml.get_object("NewBottleDialog")
        newbottledlg.destroy()

    def AddBottleFinished(self, op):
        dlgWidget = self.modaldialogxml.get_object("NewBottleDialog")
        progbar = self.modaldialogxml.get_object("NewBottleProgbar")
        GLib.source_remove(self.timeout_source)
        progbar.hide()

        if not op.exitStatus[0]:
            cxguitools.CXMessageDlg(primary=_("Could not create bottle"), secondary=op.exitStatus[1], message_type=Gtk.MessageType.ERROR, parent=self.main_window.main_window)
        bottlecollection.sharedCollection().refresh()
        dlgWidget.destroy()

    @staticmethod
    def on_destroy(_unused):
        cxguitools.toplevel_quit()


class BottleListItem(GObject.Object):

    PRIVATE_BOTTLES = 3
    MANAGED_BOTTLES = 6

    def __init__(self, key, bottle=None, markup=None):
        GObject.Object.__init__(self)

        self.key = key
        self.bottle = bottle
        self.markup = markup

        if not markup:
            self.markup = self.get_markup()

        self.bottle_name = None
        if bottle:
            self.bottle_name = bottle.name

        self.label = Gtk.Label()
        self.entry = Gtk.Entry()

        self.widget = None
        self.editing = False

    @staticmethod
    def get_bottle_sort_index(bottle):
        if bottle.is_managed:
            if bottle.is_default:
                return BottleListItem.MANAGED_BOTTLES + 1

            return BottleListItem.MANAGED_BOTTLES + 2

        if bottle.is_default:
            return BottleListItem.PRIVATE_BOTTLES + 1

        return BottleListItem.PRIVATE_BOTTLES + 2

    @staticmethod
    def compare(item_a, item_b):
        if item_a.bottle is not None:
            index_a = BottleListItem.get_bottle_sort_index(item_a.bottle)
        else:
            index_a = item_a.key

        if item_b.bottle is not None:
            index_b = BottleListItem.get_bottle_sort_index(item_b.bottle)
        else:
            index_b = item_b.key

        return  1 if index_a > index_b else \
               -1 if index_a < index_b else \
               -1 if item_a.bottle is None else \
                1 if item_b.bottle is None else \
               cxutils.cmp(item_a.bottle_name.casefold(), item_b.bottle_name.casefold())

    def set_markup(self, markup):
        if markup != self.markup:
            self.markup = markup
            self.label.set_markup(markup)

    @staticmethod
    def get_bottle_status_text(bottle):
        for status in reversed(bottle.status_overrides):
            if status not in (bottle.STATUS_INIT, bottle.STATUS_UPGRADE,
                              bottle.STATUS_READY, bottle.STATUS_DEFAULTING):
                return status
        return None

    def get_markup(self, indent="  "):
        if not self.bottle:
            return self.markup

        html_text = cxutils.html_escape(self.bottle.name)
        if self.bottle.is_default:
            html_text = '<b>' + html_text + '</b>'
        status_text = self.get_bottle_status_text(self.bottle)
        if status_text:
            html_text += (" (%s)" % cxutils.html_escape(status_text))
        return indent + html_text

    def get_widget(self):
        if self.widget:
            return self.widget

        self.label.set_markup(self.markup)
        self.label.set_halign(Gtk.Align.START)
        self.label.set_property('margin', 6)
        self.label.set_ellipsize(Pango.EllipsizeMode.END)

        self.widget = Gtk.ListBoxRow()

        if self.key in (self.PRIVATE_BOTTLES, self.MANAGED_BOTTLES):
            self.widget.set_sensitive(False)
            self.widget.set_selectable(False)
            self.widget.set_activatable(False)
            self.label.set_property('margin', 0)
            self.label.set_property('margin-top', 10)
            self.label.set_property('margin-left', 6)

        box = Gtk.Box()
        box.pack_start(self.label, True, True, 0)
        box.pack_start(self.entry, True, True, 0)

        self.entry.connect('activate', self.on_activate)
        self.entry.connect('focus-out-event', self.on_focus_out_event)
        self.entry.connect('insert-text', self.on_insert_text)

        self.widget.add(box)
        self.widget.show_all()
        self.entry.hide()

        self.widget.set_property('has-tooltip', True)
        self.widget.connect('query-tooltip', self.on_query_tooltip)

        return self.widget

    def edit(self):
        if self.editing or not self.bottle or not self.bottle.can_rename:
            return

        self.entry.set_text(self.bottle.name)
        self.entry.set_halign(Gtk.Align.START)
        self.entry.set_property('margin', 6)

        self.editing = True

        self.label.hide()
        self.entry.show()

        GLib.idle_add(self.entry.grab_focus)

    def stop_editing(self):
        if not self.editing:
            return

        self.editing = False

        old_name = self.bottle.name
        new_name = cxutils.string_to_unicode(self.entry.get_text())

        if old_name != new_name:
            self.bottle.rename(new_name, bottlecollection.sharedCollection().refresh)

        self.entry.hide()
        self.label.show()

    def on_activate(self, _widget):
        self.stop_editing()

    def on_focus_out_event(self, _widget, _event):
        self.stop_editing()

    def on_query_tooltip(self, _view, _x, _y, _keyboard_mode, tooltip):
        tooltip.set_markup(self.get_markup(''))
        return True

    @staticmethod
    def on_insert_text(widget, new_text, _length, position):
        name = widget.get_text()
        position = widget.get_position()
        name = name[:position] + new_text + name[position:]

        if not cxutils.is_valid_bottlename(name):
            widget.stop_emission_by_name('insert-text')


class SystemInfoDialog:
    def __init__(self):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("textdialog"))
        self.xml.connect_signals(self)

        self.xml.get_object("TextDialog").set_title(_("System Information"))

        self.contents = systeminfo.system_info_string()
        self.xml.get_object("TextContents").get_buffer().set_text(self.contents)

    def quit_requested(self, *_args):
        self.xml.get_object("TextDialog").destroy()
        cxguitools.toplevel_quit()

    def copy_clicked(self, _caller):
        clipboard = Gtk.Clipboard.get_default(Gdk.Display.get_default())
        clipboard.set_text(self.contents, -1)
        clipboard.store()


class UpdateCompatDBOperation(pyop.PythonOperation):
    def main(self):
        webutils.update_compatdb()


class UpdateProfilesOperation(pyop.PythonOperation):

    def __init__(self, url, parent):
        pyop.PythonOperation.__init__(self)
        self.url = url
        self.parent = parent
        self.refresh = multiprocessing.Value('b', False)

    def __unicode__(self):
        return "%s - %s" % (self.__class__.__name__, str(self.url))

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def process_main(self):
        self.refresh.value = c4profilesmanager.update_online_profiles(self.url)

    def main(self):
        worker = multiprocessing.Process(target=self.process_main, daemon=True)
        worker.start()
        worker.join()

    def finish(self):
        if self.refresh.value:
            self.parent.package_view_controller.update_profiles()

        self.parent.crossover_gui.get_object('UpdatingProfiles').hide()


class CheckRegistrationOperation(pyop.PythonOperation):
    def __init__(self, parent):
        pyop.PythonOperation.__init__(self)
        self.parent = parent
        self.is_demo = True

    def __unicode__(self):
        return "CheckRegistrationOperation"

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        license_file = os.path.join(cxutils.CX_ROOT, "etc", "license.txt")
        sig_file = os.path.join(cxutils.CX_ROOT, "etc", "license.sha256")
        if not os.path.exists(sig_file):
            sig_file = os.path.join(cxutils.CX_ROOT, "etc", "license.sig")
        (self.is_demo, _username, _date, _licenseid, _revoked) = demoutils.demo_status(license_file, sig_file)

    def finish(self):
        self.parent.is_demo = self.is_demo
        pyop.PythonOperation.finish(self)


class NewBottleOperation(pyop.PythonOperation):

    def __init__(self, bottle_name, template, new_bottle_controller, bottle_to_copy=None, publish=False):
        pyop.PythonOperation.__init__(self)
        self.exitStatus = None
        self.bottle_to_copy = bottle_to_copy
        self.new_bottle_name = bottle_name
        self.template = template
        self.publish = publish
        self.new_bottle_controller = new_bottle_controller

    def __unicode__(self):
        return "NewBottleOperation for " + self.new_bottle_name

    def enqueued(self):
        bottlecollection.sharedCollection().add_ignored_bottle(self.new_bottle_name)
        pyop.PythonOperation.enqueued(self)

    def main(self):
        if self.publish:
            self.exitStatus = bottlemanagement.publish_bottle(self.new_bottle_name, self.bottle_to_copy)
        elif self.bottle_to_copy:
            self.exitStatus = bottlemanagement.copy_bottle(self.new_bottle_name, self.bottle_to_copy)
        else:
            run_cxfixes(self.template.endswith("_64"))
            self.exitStatus = bottlemanagement.create_bottle(self.new_bottle_name, self.template)

    def finish(self):
        bottlecollection.sharedCollection().remove_ignored_bottle(self.new_bottle_name)
        self.new_bottle_controller.AddBottleFinished(self)
        pyop.PythonOperation.finish(self)


class RestoreArchiveController:
    def __init__(self, main_window, archive=None):
        self.main_window = main_window

        self.timeout_source = None

        self.modaldialogxml = Gtk.Builder()
        self.modaldialogxml.set_translation_domain("crossover")
        self.modaldialogxml.add_from_file(cxguitools.get_ui_path("bottlearchivepicker"))
        self.modaldialogxml.connect_signals(self)
        pickerdlg = self.modaldialogxml.get_object("BottleArchivePicker")
        pickerdlg.props.transient_for = main_window.main_window
        cxguitools.add_filters(pickerdlg, cxguitools.FILTERS_CXARCHIVES)

        if archive:
            pickerdlg.set_filename(archive)
            self.suggestBottleName(archive)

        progbar = self.modaldialogxml.get_object("RestoringProgbar")
        progbar.hide()
        pickerdlg.show()

    def cancel(self, _caller):
        pickerdlg = self.modaldialogxml.get_object("BottleArchivePicker")
        pickerdlg.destroy()

    def pulse(self):
        self.modaldialogxml.get_object("RestoringProgbar").pulse()
        return True

    def open(self, _caller):
        pickerdlg = self.modaldialogxml.get_object("BottleArchivePicker")
        bottleNameWidget = self.modaldialogxml.get_object("newbottlename")

        progbar = self.modaldialogxml.get_object("RestoringProgbar")

        archiveFilename = pickerdlg.get_filename()
        newBottleOp = RestoreBottleOperation(cxutils.string_to_unicode(bottleNameWidget.get_text()), archiveFilename, self)
        progbar.show()
        pickerdlg.set_sensitive(False)

        self.timeout_source = GLib.timeout_add(100, self.pulse)

        pyop.sharedOperationQueue.enqueue(newBottleOp)

    def file_selection_changed(self, _caller):
        pickerdlg = self.modaldialogxml.get_object("BottleArchivePicker")
        selectedFilename = pickerdlg.get_filename()
        self.suggestBottleName(selectedFilename)

    def suggestBottleName(self, archiveFile):
        openButton = self.modaldialogxml.get_object("openbutton")
        bottleNameWidget = self.modaldialogxml.get_object("newbottlename")
        if archiveFile and not os.path.isdir(archiveFile):
            root = os.path.splitext(os.path.basename(archiveFile))[0]
            bottleNameWidget.set_text(bottlequery.unique_bottle_name(root))
        else:
            bottleNameWidget.set_text("")
            openButton.set_sensitive(False)

    def bottle_name_delete_text(self, caller, start, stop):
        name = caller.get_text()
        name = name[:start] + name[stop:]
        if not cxutils.is_valid_bottlename(name):
            caller.stop_emission_by_name("delete-text")
        else:
            okButton = self.modaldialogxml.get_object("openbutton")
            okButton.set_sensitive(bottlequery.is_valid_new_bottle_name(name))

    def bottle_name_insert_text(self, caller, new_text, _length, _user_data):
        name = caller.get_text()
        position = caller.get_position()
        name = name[:position] + new_text + name[position:]
        if not cxutils.is_valid_bottlename(name):
            caller.stop_emission_by_name("insert-text")
        else:
            okButton = self.modaldialogxml.get_object("openbutton")
            okButton.set_sensitive(bottlequery.is_valid_new_bottle_name(name))

    def RestoreBottleFinished(self, op):
        pickerdlg = self.modaldialogxml.get_object("BottleArchivePicker")
        progbar = self.modaldialogxml.get_object("RestoringProgbar")
        GLib.source_remove(self.timeout_source)

        progbar.hide()
        pickerdlg.set_sensitive(True)

        if not op.exitStatus[0]:
            cxguitools.CXMessageDlg(primary=_("Could not restore bottle"), secondary=op.exitStatus[1], message_type=Gtk.MessageType.ERROR, parent=self.main_window.main_window)
        else:
            bottlecollection.sharedCollection().refresh()
            pickerdlg.destroy()


class RestoreBottleOperation(pyop.PythonOperation):

    def __init__(self, bottle_name, archive_file, controller_window):
        pyop.PythonOperation.__init__(self)
        self.exitStatus = None
        self.archive_file = archive_file
        self.bottle_name = bottle_name
        self.controller_window = controller_window

    def __unicode__(self):
        return "RestoreBottleOperation for " + self.bottle_name

    def enqueued(self):
        bottlecollection.sharedCollection().add_ignored_bottle(self.bottle_name)
        pyop.PythonOperation.enqueued(self)

    def main(self):
        run_cxfixes()
        self.exitStatus = bottlemanagement.restore_bottle(self.bottle_name, self.archive_file)

    def finish(self):
        bottlecollection.sharedCollection().remove_ignored_bottle(self.bottle_name)
        self.controller_window.RestoreBottleFinished(self)
        pyop.PythonOperation.finish(self)


class PublishBottleController:

    def __init__(self, main_window, orig_bottle_name=None):
        self.main_window = main_window

        self.timeout_source = None

        self.modaldialogxml = Gtk.Builder()
        self.modaldialogxml.set_translation_domain("crossover")
        self.modaldialogxml.add_from_file(cxguitools.get_ui_path("publishbottledialog"))
        self.modaldialogxml.connect_signals(self)
        publishbottledlg = self.modaldialogxml.get_object("PublishBottleDialog")
        publishbottledlg.props.transient_for = main_window.main_window

        self.orig_bottle_name = orig_bottle_name

        publishedNameEntry = self.modaldialogxml.get_object("publishedNameEntry")
        publishedNameEntry.set_text(_("published_%(bottlename)s") % {'bottlename': orig_bottle_name})

        progbar = self.modaldialogxml.get_object("publishBottleProgbar")
        progbar.hide()
        publishbottledlg.show()

    def pulse(self):
        self.modaldialogxml.get_object("publishBottleProgbar").pulse()
        return True

    def publish_bottle_ok(self, _caller):
        dlgWidget = self.modaldialogxml.get_object("PublishBottleDialog")
        bottleNameWidget = self.modaldialogxml.get_object("publishedNameEntry")
        progbar = self.modaldialogxml.get_object("publishBottleProgbar")

        name = cxutils.string_to_unicode(bottleNameWidget.get_text())

        if not bottlequery.is_valid_new_bottle_name(name):
            cxguitools.CXMessageDlg(primary=_("Could not publish bottle"), secondary=_("A bottle with this name already exists."), message_type=Gtk.MessageType.ERROR, parent=self.main_window.main_window)
            return

        newBottleOp = NewBottleOperation(name, "", self, self.orig_bottle_name, True)
        progbar.show()
        dlgWidget.set_sensitive(False)

        self.timeout_source = GLib.timeout_add(100, self.pulse)

        pyop.sharedOperationQueue.enqueue(newBottleOp)

    def publish_bottle_cancel(self, _caller):
        publishbottledlg = self.modaldialogxml.get_object("PublishBottleDialog")
        publishbottledlg.destroy()

    def AddBottleFinished(self, op):
        dlgWidget = self.modaldialogxml.get_object("PublishBottleDialog")
        progbar = self.modaldialogxml.get_object("publishBottleProgbar")
        GLib.source_remove(self.timeout_source)

        progbar.hide()
        dlgWidget.set_sensitive(True)

        if not op.exitStatus[0]:
            cxguitools.CXMessageDlg(primary=_("Could not publish bottle"), secondary=op.exitStatus[1], message_type=Gtk.MessageType.ERROR, parent=self.main_window.main_window)
        else:
            bottlecollection.sharedCollection().refresh()
            dlgWidget.destroy()


class UpgradeBottleOperation(pyop.PythonOperation):

    def __init__(self, bottle):
        pyop.PythonOperation.__init__(self)
        self.bottle = bottle

    def __unicode__(self):
        return "UpgradeBottleOperation for " + self.bottle.name

    def enqueued(self):
        self.bottle.add_status_override(self.bottle.STATUS_UPGRADING)
        pyop.PythonOperation.enqueued(self)

    def main(self):
        wine = os.path.join(cxutils.CX_ROOT, "bin", "wine")

        if cxproduct.is_root_install():
            cxsu = os.path.join(cxutils.CX_ROOT, "bin", "cxsu")
            cxsu_args = [cxsu, '--ignore-home']
        else:
            cxsu_args = []

        args = cxsu_args + [wine, "--bottle", self.bottle.name,
                            "--scope", "managed", "--ux-app", "true"]
        cxutils.run(args)

    def finish(self):
        self.bottle.refresh_up_to_date()
        self.bottle.remove_status_override(self.bottle.STATUS_UPGRADING)
        pyop.PythonOperation.finish(self)
