"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
const assert = require("assert");
const lsp = require("vscode-languageserver-types");
const config_1 = require("../../config");
const removeLinkDefinition_1 = require("../../languageFeatures/codeActions/removeLinkDefinition");
const diagnostics_1 = require("../../languageFeatures/diagnostics");
const documentLinks_1 = require("../../languageFeatures/documentLinks");
const tableOfContents_1 = require("../../tableOfContents");
const range_1 = require("../../types/range");
const cancellation_1 = require("../../util/cancellation");
const engine_1 = require("../engine");
const inMemoryDocument_1 = require("../inMemoryDocument");
const inMemoryWorkspace_1 = require("../inMemoryWorkspace");
const nulLogging_1 = require("../nulLogging");
const util_1 = require("../util");
async function getActions(store, doc, pos) {
    const workspace = store.add(new inMemoryWorkspace_1.InMemoryWorkspace([doc]));
    const engine = (0, engine_1.createNewMarkdownEngine)();
    const config = (0, config_1.getLsConfiguration)({});
    const tocProvider = store.add(new tableOfContents_1.MdTableOfContentsProvider(engine, workspace, nulLogging_1.nulLogger));
    const linkProvider = store.add(new documentLinks_1.MdLinkProvider(config, engine, workspace, tocProvider, nulLogging_1.nulLogger));
    const computer = new diagnostics_1.DiagnosticComputer(config, workspace, linkProvider, tocProvider, nulLogging_1.nulLogger);
    const provider = new removeLinkDefinition_1.MdRemoveLinkDefinitionCodeActionProvider();
    return Array.from(provider.getActions(doc, (0, range_1.makeRange)(pos, pos), lsp.CodeActionContext.create((await computer.compute(doc, util_1.defaultDiagnosticsOptions, cancellation_1.noopToken)).diagnostics, undefined, undefined)));
}
suite('Remove link definition code action', () => {
    test('Should not return code action when not on link', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`test`));
        const actions = await getActions(store, doc, { line: 0, character: 3 });
        assert.strictEqual(actions.length, 0);
    }));
    test('Should not return action when not on unused definition', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com`, ``, `more text`));
        const actions = await getActions(store, doc, { line: 0, character: 3 });
        assert.strictEqual(actions.length, 0);
    }));
    test('Should return when on unused definition', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com`, ``, `more text`));
        const actions = await getActions(store, doc, { line: 2, character: 3 });
        assert.strictEqual(actions.length, 1);
        const newContent = (0, util_1.applyActionEdit)(doc, actions[0]);
        assert.strictEqual(newContent, (0, util_1.joinLines)(`text`, ``, ``, `more text`));
    }));
    test('Should remove entire line instead of leaving blank line', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`[a]: http://example.com "title"`, `[b]: http://example.com/b "title2"`, `[c]: http://example.com/c "title3"`));
        const actions = await getActions(store, doc, { line: 1, character: 3 });
        assert.strictEqual(actions.length, 1);
        const newContent = (0, util_1.applyActionEdit)(doc, actions[0]);
        assert.strictEqual(newContent, (0, util_1.joinLines)(`[a]: http://example.com "title"`, `[c]: http://example.com/c "title3"`));
    }));
    test('Should return when on unused definition with title', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com "title"`, ``, `more text`));
        const actions = await getActions(store, doc, { line: 2, character: 3 });
        assert.strictEqual(actions.length, 1);
        const newContent = (0, util_1.applyActionEdit)(doc, actions[0]);
        assert.strictEqual(newContent, (0, util_1.joinLines)(`text`, ``, ``, `more text`));
    }));
    test('Should return action when on duplicate definition', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com/first`, `[def]: http://example.com/second`, ``, `more text`));
        {
            const actions = await getActions(store, doc, { line: 2, character: 3 });
            assert.strictEqual(actions.length, 1);
            const newContent = (0, util_1.applyActionEdit)(doc, actions[0]);
            assert.strictEqual(newContent, (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com/second`, ``, `more text`));
        }
        {
            const actions = await getActions(store, doc, { line: 3, character: 3 });
            assert.strictEqual(actions.length, 1);
            const newContent = (0, util_1.applyActionEdit)(doc, actions[0]);
            assert.strictEqual(newContent, (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com/first`, ``, `more text`));
        }
    }));
    test('Should prefer unused code action if link definition is both unused and duplicated', (0, util_1.withStore)(async (store) => {
        const doc = new inMemoryDocument_1.InMemoryDocument((0, util_1.workspacePath)('test.md'), (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com "title"`, `[def]: http://example.com/other "title2"`, ``, `more text`));
        const actions = await getActions(store, doc, { line: 2, character: 3 });
        assert.strictEqual(actions.length, 1);
        const newContent = (0, util_1.applyActionEdit)(doc, actions[0]);
        assert.strictEqual(newContent, (0, util_1.joinLines)(`text`, ``, `[def]: http://example.com/other "title2"`, ``, `more text`));
    }));
});
//# sourceMappingURL=removeLinkDefinition.test.js.map