#!/bin/bash

# Test package that computes default database settings

TMP_DIR="${TMPDIR}"

test -z "${LIB_DIR}" && LIB_DIR=lib

SCRIPT_BASE=`basename $0 | sed "s/_[0-9]*\.sh$//; s/-/\//g"`

BASENAME=$(basename $0 .pm)

#BEGIN DEPEND------------------------------------------------------------------

INPUT_SCRIPT="${LIB_DIR}/${SCRIPT_BASE}"

#END DEPEND--------------------------------------------------------------------

set -ue
## set -x

test -z "${TMP_DIR}" && TMP_DIR="/tmp"
TMP_DIR="${TMP_DIR}/tmp-${BASENAME}-$$"
mkdir "${TMP_DIR}"

# To make the trap portable between bash and dash, we need to trap
# "signal" 0 ("EXIT") and rely on it for the cleanup:
## trap "rm -rf '${TMP_DIR}'" 0 1 2 3 15
trap "rm -rf '${TMP_DIR}'" EXIT
trap "exit 1" HUP INT QUIT TERM

TMP_DB_MAIN="${TMP_DIR}/test4fk.db"

sqlite3 ${TMP_DB_MAIN} <<END
    create table 'data' (
        id integer primary key autoincrement,
        a integer,
        b integer
    );
    create view sum as select a, b, a + b from 'data';

    create table child (
        id integer primary key autoincrement,
        data_id integer references data(id),
        description text
    );
    create view description as select a, b, description
        from 'data' join child on data.id = data_id;
END

export TMP_DB_MAIN

perl <<'END'
#! /usr/bin/env perl

use strict;
use warnings;
use utf8;
use Data::Dumper;

use lib 'lib';
use Database;
use Database::View;
use RestfulDB::JSONAPI qw( jsonapi2data );

binmode( STDOUT, "utf8" );

$Data::Dumper::Sortkeys = 1;
$Data::Dumper::Indent   = 1;

my $db_settings = {
    content_db => {
        DB => $ENV{TMP_DB_MAIN},
        engine => 'SQLite'            
    }
};

my $db = Database->new( $db_settings );
$db->connect();

do {
    my $json = '{
        "data" : {
            "attributes" : {
                "a" : 1,
                "b" : 2
            },
            "type" : "sum"
        }
    }';

    my $data = jsonapi2data( $db, $json );
    $db->modify_record_descriptions( $data );
};

do {
    my $json = '{
        "data" : {
            "attributes" : {
                "a" : 1,
                "description" : "ABC"
            },
            "type" : "description"
        }
    }';

    my $data = jsonapi2data( $db, $json );
    $db->modify_record_descriptions( $data );
};
END

sqlite3 ${TMP_DB_MAIN} -nullvalue NULL 'select * from sum'
sqlite3 ${TMP_DB_MAIN} -nullvalue NULL 'select * from description'

# Temporary directory will be deleted by the trap.
