// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef EXTENSIONS_BROWSER_EXTENSION_FUNCTION_HISTOGRAM_VALUE_H_
#define EXTENSIONS_BROWSER_EXTENSION_FUNCTION_HISTOGRAM_VALUE_H_

namespace extensions {
namespace functions {

// Short version:
//  *Never* reorder or delete entries in the |HistogramValue| enumeration.
//  When creating a new extension function, add a new entry at the end of the
//  enum, just prior to ENUM_BOUNDARY.
//
// Long version:
//  This enumeration is used to associate a unique integer value to each
//  extension function so that their usage can be recorded in histogram charts.
//  Given we want the values recorded in the these charts to remain stable over
//  time for comparison purposes, once an entry has been added to the
//  enumeration, it should never be removed or moved to another spot in the
//  enum.
//
//  Here are instructions how to manage entries depending on what you are trying
//  to achieve.
//
//  1) Creating a new extension function:
//
//      Add a new entry at the end of the |HistogramValue| enum. The name of the
//      entry should follow this algorithm:
//      a) Take the string value passed as first argument to
//         DECLARE_EXTENSION_FUNCTION.
//      b) Replace '.' with '_'.
//      c) Make all letters uppercase.
//
//      Example: "tabs.create" -> TABS_CREATE
//
//  2) Deleting an existing function:
//
//      Given an existing entry should *never* be removed from this enumeration,
//      it is recommended to add a "DELETED_" prefix to the existing entry.
//
//  3) Renaming an existing function:
//
//      There are 2 options, depending if you want to keep accruing data in the
//      *existing* histogram stream or in a *new* one.
//
//      a) If you want keep recording usages of the extension function in the
//         *existing* histogram stream, simply rename the enum entry to match
//         the new extension function name, following the same naming rule as
//         mentioned in 1). The enum entry will keep the same underlying integer
//         value, so the same histogram stream will be used for recording
//         usages.
//
//      b) If you want start recording usages of the extension function to in a
//         *new* histogram stream, follow the instructions in step 1) and 2)
//         above. This will effectively deprecate the old histogram stream and
//         creates a new one for the new function name.
//
enum HistogramValue {
  UNKNOWN = 0,
  WEBNAVIGATION_GETALLFRAMES = 1,
  BROWSINGDATA_REMOVEWEBSQL = 2,
  ALARMS_CREATE = 3,
  FILEMANAGERPRIVATEINTERNAL_REMOVEFILEWATCH = 4,
  COOKIES_GET = 5,
  FONTSETTINGS_GETMINIMUMFONTSIZE = 6,
  CHROMEOSINFOPRIVATE_GET = 7,
  BOOKMARKMANAGERPRIVATE_CUT = 8,
  TABS_CAPTUREVISIBLETAB = 9,
  MANAGEMENT_SETENABLED = 10,
  HISTORY_DELETEALL = 11,
  STORAGE_GET = 12,
  SOCKET_SETKEEPALIVE = 13,
  DOWNLOADS_CANCEL = 14,
  BOOKMARKS_CREATE = 15,
  BOOKMARKS_UPDATE = 16,
  FILEMANAGERPRIVATE_GETDRIVEFILES = 17,
  TERMINALPRIVATE_ONTERMINALRESIZE = 18,
  DELETED_FILEMANAGERPRIVATE_REQUESTDIRECTORYREFRESH = 19,
  BLUETOOTH_GETADAPTERSTATE = 20,
  DELETED_FILEMANAGERPRIVATEINTERNAL_CANCELFILETRANSFERS = 21,
  FILEMANAGERPRIVATEINTERNAL_PINDRIVEFILE = 22,
  SOCKET_WRITE = 23,
  OMNIBOX_SETDEFAULTSUGGESTION = 24,
  TTS_SPEAK = 25,
  WALLPAPERPRIVATE_RESTOREMINIMIZEDWINDOWS = 26,
  BROWSINGDATA_REMOVEHISTORY = 27,
  DELETED_FILEMANAGERPRIVATE_ISFULLSCREEN = 28,
  AUTOTESTPRIVATE_LOGOUT = 29,
  EXPERIMENTAL_HISTORY_GETMOSTVISITED = 30,
  DELETED_BLUETOOTH_DISCONNECT = 31,
  DELETED_BLUETOOTH_SETOUTOFBANDPAIRINGDATA = 32,
  BOOKMARKMANAGERPRIVATE_CANPASTE = 33,
  AUTOTESTPRIVATE_RESTART = 34,
  USB_CLAIMINTERFACE = 35,
  MEDIAPLAYERPRIVATE_SETWINDOWHEIGHT = 36,
  DELETED_EXPERIMENTAL_PROCESSES_GETPROCESSINFO = 37,
  HISTORY_GETVISITS = 38,
  SOCKET_BIND = 39,
  TABS_MOVE = 40,
  SOCKET_DISCONNECT = 41,
  FILESYSTEM_GETWRITABLEENTRY = 42,
  SYNCFILESYSTEM_REQUESTFILESYSTEM = 43,
  COMMANDS_GETALL = 44,
  EXPERIMENTAL_DISCOVERY_REMOVESUGGESTION = 45,
  VIRTUALKEYBOARDPRIVATE_SENDKEYEVENT = 46,
  BOOKMARKMANAGERPRIVATE_GETSUBTREE = 47,
  DELETED_EXPERIMENTAL_RLZ_RECORDPRODUCTEVENT = 48,
  BOOKMARKS_GETRECENT = 49,
  APP_CURRENTWINDOWINTERNAL_SETBOUNDS = 50,
  CLOUDPRINTPRIVATE_SETUPCONNECTOR = 51,
  SERIAL_SETCONTROLSIGNALS = 52,
  DELETED_FILEMANAGERPRIVATE_SETLASTMODIFIED = 53,
  IDLE_SETDETECTIONINTERVAL = 54,
  FILEMANAGERPRIVATEINTERNAL_GETFILETASKS = 55,
  WEBSTOREPRIVATE_GETSTORELOGIN = 56,
  SYSTEMPRIVATE_GETINCOGNITOMODEAVAILABILITY = 57,
  IDLTEST_SENDARRAYBUFFERVIEW = 58,
  SOCKET_SETNODELAY = 59,
  APP_CURRENTWINDOWINTERNAL_SHOW = 60,
  WEBSTOREPRIVATE_GETBROWSERLOGIN = 61,
  EXPERIMENTAL_IDENTITY_GETAUTHTOKEN = 62,
  DELETED_SYSTEMINFO_DISPLAY_GETDISPLAYINFO = 63,
  BROWSINGDATA_REMOVEPLUGINDATA = 64,
  SOCKET_LISTEN = 65,
  MEDIAGALLERIES_GETMEDIAFILESYSTEMS = 66,
  DOWNLOADS_OPEN = 67,
  TABS_EXECUTESCRIPT = 68,
  SYNCFILESYSTEM_GETUSAGEANDQUOTA = 69,
  INPUTMETHODPRIVATE_GET = 70,
  USB_CLOSEDEVICE = 71,
  TTS_STOP = 72,
  DELETED_SERIAL_GETPORTS = 73,
  DELETED_FILEMANAGERPRIVATE_CLEARDRIVECACHE = 74,
  SERIAL_GETCONTROLSIGNALS = 75,
  DELETED_DEVELOPERPRIVATE_ENABLE = 76,
  FILEMANAGERPRIVATEINTERNAL_GETENTRYPROPERTIES = 77,
  USB_FINDDEVICES = 78,
  BOOKMARKMANAGERPRIVATE_DROP = 79,
  DELETED_FILEMANAGERPRIVATE_GETFILETRANSFERS = 80,
  INPUT_IME_SETMENUITEMS = 81,
  BOOKMARKS_EXPORT = 82,
  HISTORY_SEARCH = 83,
  TTSENGINE_SENDTTSEVENT = 84,
  ACCESSIBILITY_PRIVATE_GETALERTSFORTAB = 85,
  BOOKMARKS_IMPORT = 86,
  SYNCFILESYSTEM_DELETEFILESYSTEM = 87,
  DEBUGGER_SENDCOMMAND = 88,
  DEBUGGER_DETACH = 89,
  METRICSPRIVATE_RECORDSMALLCOUNT = 90,
  APP_CURRENTWINDOWINTERNAL_MINIMIZE = 91,
  DEVELOPERPRIVATE_AUTOUPDATE = 92,
  DNS_RESOLVE = 93,
  DELETED_EXPERIMENTAL_SYSTEMINFO_MEMORY_GET = 94,
  HISTORY_ADDURL = 95,
  TABS_GET = 96,
  BROWSERACTION_SETBADGETEXT = 97,
  TABS_RELOAD = 98,
  WINDOWS_CREATE = 99,
  DEVELOPERPRIVATE_LOADUNPACKED = 100,
  DELETED_DOWNLOADS_SETDESTINATION = 101,
  DELETED_EXPERIMENTAL_PROCESSES_GETPROCESSIDFORTAB = 102,
  BOOKMARKS_GETCHILDREN = 103,
  BROWSERACTION_GETTITLE = 104,
  TERMINALPRIVATE_OPENTERMINALPROCESS = 105,
  DELETED_SERIAL_CLOSE = 106,
  CONTEXTMENUS_REMOVE = 107,
  FILESYSTEM_REQUESTFILESYSTEM = 108,
  ECHOPRIVATE_GETREGISTRATIONCODE = 109,
  TABS_GETCURRENT = 110,
  FONTSETTINGS_CLEARDEFAULTFIXEDFONTSIZE = 111,
  MEDIAPLAYERPRIVATE_CLOSEWINDOW = 112,
  WEBREQUESTINTERNAL_ADDEVENTLISTENER = 113,
  CLOUDPRINTPRIVATE_GETPRINTERS = 114,
  STORAGE_SET = 115,
  FONTSETTINGS_GETDEFAULTFONTSIZE = 116,
  EXTENSION_SETUPDATEURLDATA = 117,
  DELETED_SERIAL_WRITE = 118,
  IDLE_QUERYSTATE = 119,
  DELETED_EXPERIMENTAL_RLZ_GETACCESSPOINTRLZ = 120,
  WEBSTOREPRIVATE_SETSTORELOGIN = 121,
  DELETED_PAGEACTIONS_ENABLEFORTAB = 122,
  COOKIES_SET = 123,
  CONTENTSETTINGS_SET = 124,
  CONTEXTMENUS_REMOVEALL = 125,
  TABS_INSERTCSS = 126,
  WEBREQUEST_HANDLERBEHAVIORCHANGED = 127,
  INPUT_IME_SETCURSORPOSITION = 128,
  OMNIBOX_SENDSUGGESTIONS = 129,
  SYSTEMINDICATOR_ENABLE = 130,
  EVENTS_GETRULES = 131,
  BOOKMARKMANAGERPRIVATE_COPY = 132,
  SOCKET_RECVFROM = 133,
  TABS_GETALLINWINDOW = 134,
  CONTEXTMENUS_UPDATE = 135,
  BOOKMARKS_SEARCH = 136,
  EXPERIMENTAL_APP_CLEARALLNOTIFICATIONS = 137,
  DELETED_BLUETOOTH_GETLOCALOUTOFBANDPAIRINGDATA = 138,
  SYSTEMPRIVATE_GETUPDATESTATUS = 139,
  FONTSETTINGS_CLEARMINIMUMFONTSIZE = 140,
  DELETED_FILEMANAGERPRIVATE_GETFILELOCATIONS = 141,
  EXPERIMENTAL_DISCOVERY_SUGGEST = 142,
  FILEMANAGERPRIVATEINTERNAL_SETDEFAULTTASK = 143,
  BROWSERACTION_GETBADGETEXT = 144,
  APP_CURRENTWINDOWINTERNAL_HIDE = 145,
  SOCKET_CONNECT = 146,
  BOOKMARKS_GETSUBTREE = 147,
  HISTORY_DELETEURL = 148,
  DELETED_EXPERIMENTAL_MEDIAGALLERIES_ASSEMBLEMEDIAFILE = 149,
  BOOKMARKMANAGERPRIVATE_STARTDRAG = 150,
  BROWSINGDATA_REMOVEPASSWORDS = 151,
  DELETED_DOWNLOADS_DRAG = 152,
  INPUT_IME_SETCOMPOSITION = 153,
  METRICSPRIVATE_RECORDUSERACTION = 154,
  USB_RELEASEINTERFACE = 155,
  PAGEACTION_GETPOPUP = 156,
  DELETED_SCRIPTBADGE_GETATTENTION = 157,
  FONTSETTINGS_GETFONTLIST = 158,
  PERMISSIONS_CONTAINS = 159,
  DELETED_SCRIPTBADGE_GETPOPUP = 160,
  ACCESSIBILITY_PRIVATE_GETFOCUSEDCONTROL = 161,
  DEVELOPERPRIVATE_GETSTRINGS = 162,
  METRICSPRIVATE_RECORDMEDIUMCOUNT = 163,
  MANAGEMENT_GET = 164,
  PERMISSIONS_GETALL = 165,
  DOWNLOADS_SHOW = 166,
  DELETED_EXPERIMENTAL_RLZ_CLEARPRODUCTSTATE = 167,
  TABS_REMOVE = 168,
  MANAGEMENT_GETPERMISSIONWARNINGSBYID = 169,
  WINDOWS_GET = 170,
  FILEMANAGERPRIVATEINTERNAL_EXECUTETASK = 171,
  TTS_GETVOICES = 172,
  MANAGEMENT_GETALL = 173,
  MANAGEMENT_GETPERMISSIONWARNINGSBYMANIFEST = 174,
  APP_CURRENTWINDOWINTERNAL_CLEARATTENTION = 175,
  AUTOTESTPRIVATE_SHUTDOWN = 176,
  FONTSETTINGS_CLEARDEFAULTFONTSIZE = 177,
  BOOKMARKS_GETTREE = 178,
  FILEMANAGERPRIVATE_SELECTFILES = 179,
  RUNTIME_GETBACKGROUNDPAGE = 180,
  DELETED_EXPERIMENTAL_RECORD_REPLAYURLS = 181,
  WEBSTOREPRIVATE_COMPLETEINSTALL = 182,
  DELETED_EXPERIMENTAL_SPEECHINPUT_START = 183,
  COOKIES_GETALL = 184,
  DOWNLOADS_GETFILEICON = 185,
  PAGEACTION_GETTITLE = 186,
  BROWSINGDATA_REMOVE = 187,
  DELETED_SERIAL_OPEN = 188,
  FILESYSTEM_GETDISPLAYPATH = 189,
  FILEMANAGERPRIVATE_FORMATVOLUME = 190,
  BOOKMARKS_GET = 191,
  DELETED_MANAGEDMODEPRIVATE_GET = 192,
  ALARMS_CLEAR = 193,
  SYNCFILESYSTEM_GETFILESYNCSTATUS = 194,
  SOCKET_GETINFO = 195,
  WEBSTOREPRIVATE_INSTALLBUNDLE = 196,
  BROWSERACTION_ENABLE = 197,
  METRICSPRIVATE_RECORDMEDIUMTIME = 198,
  PAGEACTION_SETTITLE = 199,
  CLOUDPRINTPRIVATE_GETHOSTNAME = 200,
  CONTENTSETTINGS_GETRESOURCEIDENTIFIERS = 201,
  SOCKET_CREATE = 202,
  DEVELOPERPRIVATE_RELOAD = 203,
  FILEMANAGERPRIVATE_GETVOLUMEMETADATALIST = 204,
  APP_RUNTIME_POSTINTENTRESPONSE = 205,
  DELETED_MANAGEDMODEPRIVATE_SETPOLICY = 206,
  WEBSTOREPRIVATE_BEGININSTALLWITHMANIFEST3 = 207,
  WALLPAPERPRIVATE_SETWALLPAPER = 208,
  USB_CONTROLTRANSFER = 209,
  DELETED_EXPERIMENTAL_SPEECHINPUT_STOP = 210,
  USB_BULKTRANSFER = 211,
  DELETED_FILEMANAGERPRIVATE_GETVOLUMEMETADATA = 212,
  PAGECAPTURE_SAVEASMHTML = 213,
  EXTENSION_ISALLOWEDINCOGNITOACCESS = 214,
  BROWSINGDATA_REMOVEAPPCACHE = 215,
  APP_CURRENTWINDOWINTERNAL_DRAWATTENTION = 216,
  METRICSPRIVATE_RECORDCOUNT = 217,
  USB_INTERRUPTTRANSFER = 218,
  TYPES_CHROMESETTING_CLEAR = 219,
  INPUT_IME_COMMITTEXT = 220,
  IDLTEST_SENDARRAYBUFFER = 221,
  WALLPAPERPRIVATE_SETWALLPAPERIFEXISTS = 222,
  SOCKET_ACCEPT = 223,
  WEBNAVIGATION_GETFRAME = 224,
  EXPERIMENTAL_POWER_RELEASEKEEPAWAKE = 225,
  APP_CURRENTWINDOWINTERNAL_SETICON = 226,
  DELETED_PUSHMESSAGING_GETCHANNELID = 227,
  DELETED_EXPERIMENTAL_INFOBARS_SHOW = 228,
  INPUT_IME_SETCANDIDATEWINDOWPROPERTIES = 229,
  METRICSPRIVATE_RECORDPERCENTAGE = 230,
  TYPES_CHROMESETTING_GET = 231,
  WINDOWS_GETLASTFOCUSED = 232,
  DELETED_MANAGEDMODEPRIVATE_GETPOLICY = 233,
  STORAGE_CLEAR = 234,
  STORAGE_GETBYTESINUSE = 235,
  TABS_QUERY = 236,
  PAGEACTION_SETPOPUP = 237,
  DEVELOPERPRIVATE_INSPECT = 238,
  DOWNLOADS_SEARCH = 239,
  FONTSETTINGS_CLEARFONT = 240,
  WINDOWS_UPDATE = 241,
  DELETED_BOOKMARKMANAGERPRIVATE_CANOPENNEWWINDOWS = 242,
  SERIAL_FLUSH = 243,
  BROWSERACTION_SETTITLE = 244,
  BOOKMARKMANAGERPRIVATE_CANEDIT = 245,
  WALLPAPERPRIVATE_SETCUSTOMWALLPAPER = 246,
  BOOKMARKS_REMOVE = 247,
  INPUT_IME_SETCANDIDATES = 248,
  TERMINALPRIVATE_CLOSETERMINALPROCESS = 249,
  HISTORY_DELETERANGE = 250,
  IDLTEST_GETARRAYBUFFER = 251,
  TERMINALPRIVATE_SENDINPUT = 252,
  TABS_HIGHLIGHT = 253,
  BLUETOOTH_STARTDISCOVERY = 254,
  FILEMANAGERPRIVATE_SELECTFILE = 255,
  WINDOWS_GETCURRENT = 256,
  DEBUGGER_ATTACH = 257,
  WALLPAPERPRIVATE_SAVETHUMBNAIL = 258,
  INPUT_IME_KEYEVENTHANDLED = 259,
  FONTSETTINGS_SETDEFAULTFONTSIZE = 260,
  RUNTIME_REQUESTUPDATECHECK = 261,
  PAGEACTION_SETICON = 262,
  BROWSERACTION_SETBADGEBACKGROUNDCOLOR = 263,
  DEVELOPERPRIVATE_GETITEMSINFO = 264,
  BLUETOOTH_STOPDISCOVERY = 265,
  COOKIES_REMOVE = 266,
  DELETED_EXPERIMENTAL_RLZ_SENDFINANCIALPING = 267,
  TABCAPTURE_GETCAPTUREDTABS = 268,
  WINDOWS_REMOVE = 269,
  WALLPAPERPRIVATE_GETOFFLINEWALLPAPERLIST = 270,
  BROWSERACTION_GETBADGEBACKGROUNDCOLOR = 271,
  DELETED_PAGEACTIONS_DISABLEFORTAB = 272,
  DELETED_DEVELOPERPRIVATE_ALLOWFILEACCESS = 273,
  FILEMANAGERPRIVATE_REMOVEMOUNT = 274,
  DELETED_BLUETOOTH_CONNECT = 275,
  TABCAPTURE_CAPTURE = 276,
  NOTIFICATIONS_CREATE = 277,
  TABS_DUPLICATE = 278,
  DELETED_BLUETOOTH_WRITE = 279,
  PAGEACTION_SHOW = 280,
  WALLPAPERPRIVATE_GETTHUMBNAIL = 281,
  DOWNLOADS_PAUSE = 282,
  PERMISSIONS_REQUEST = 283,
  TOPSITES_GET = 284,
  BROWSINGDATA_REMOVEDOWNLOADS = 285,
  BROWSINGDATA_REMOVELOCALSTORAGE = 286,
  FILEBROWSERHANDLERINTERNAL_SELECTFILE = 287,
  INPUT_IME_UPDATEMENUITEMS = 288,
  FILEMANAGERPRIVATE_GETSTRINGS = 289,
  CONTENTSETTINGS_GET = 290,
  FONTSETTINGS_SETDEFAULTFIXEDFONTSIZE = 291,
  EXPERIMENTAL_APP_NOTIFY = 292,
  METRICSPRIVATE_RECORDLONGTIME = 293,
  SOCKET_READ = 294,
  DELETED_EXPERIMENTAL_PROCESSES_TERMINATE = 295,
  METRICSPRIVATE_RECORDTIME = 296,
  DELETED_BOOKMARKMANAGERPRIVATE_GETSTRINGS = 297,
  USB_ISOCHRONOUSTRANSFER = 298,
  PERMISSIONS_REMOVE = 299,
  MANAGEMENT_UNINSTALL = 300,
  I18N_GETACCEPTLANGUAGES = 301,
  MANAGEMENT_LAUNCHAPP = 302,
  INPUT_IME_CLEARCOMPOSITION = 303,
  ALARMS_GETALL = 304,
  DELETED_DIAL_DISCOVERNOW = 305,
  TYPES_CHROMESETTING_SET = 306,
  BROWSERACTION_SETICON = 307,
  ACCESSIBILITY_PRIVATE_SETACCESSIBILITYENABLED = 308,
  DELETED_FILEMANAGERPRIVATE_VIEWFILES = 309,
  DELETED_BLUETOOTH_GETSERVICES = 310,
  TABS_UPDATE = 311,
  BROWSINGDATA_REMOVEFORMDATA = 312,
  DELETED_FILEMANAGERPRIVATE_RELOADDRIVE = 313,
  ALARMS_GET = 314,
  BROWSINGDATA_REMOVEINDEXEDDB = 315,
  FILEMANAGERPRIVATEINTERNAL_ADDFILEWATCH = 316,
  CONTENTSETTINGS_CLEAR = 317,
  FILEMANAGERPRIVATE_GETPREFERENCES = 318,
  BOOKMARKMANAGERPRIVATE_PASTE = 319,
  FILESYSTEM_ISWRITABLEENTRY = 320,
  USB_SETINTERFACEALTERNATESETTING = 321,
  FONTSETTINGS_SETMINIMUMFONTSIZE = 322,
  BROWSERACTION_GETPOPUP = 323,
  SOCKET_DESTROY = 324,
  BLUETOOTH_GETDEVICES = 325,
  ALARMS_CLEARALL = 326,
  FONTSETTINGS_GETDEFAULTFIXEDFONTSIZE = 327,
  FILEMANAGERPRIVATEINTERNAL_ZIPSELECTION = 328,
  SYSTEMINDICATOR_DISABLE = 329,
  DELETED_SCRIPTBADGE_SETPOPUP = 330,
  EXTENSION_ISALLOWEDFILESCHEMEACCESS = 331,
  EXPERIMENTAL_IDENTITY_LAUNCHWEBAUTHFLOW = 332,
  FILEMANAGERPRIVATE_GETDRIVECONNECTIONSTATE = 333,
  TABS_DETECTLANGUAGE = 334,
  METRICSPRIVATE_RECORDVALUE = 335,
  BOOKMARKMANAGERPRIVATE_SORTCHILDREN = 336,
  DELETED_SERIAL_READ = 337,
  APP_CURRENTWINDOWINTERNAL_MAXIMIZE = 338,
  EXPERIMENTAL_DISCOVERY_CLEARALLSUGGESTIONS = 339,
  DELETED_MANAGEDMODEPRIVATE_ENTER = 340,
  DELETED_FILEMANAGERPRIVATE_TRANSFERFILE = 341,
  BROWSERACTION_SETPOPUP = 342,
  TABS_GETSELECTED = 343,
  FONTSETTINGS_GETFONT = 344,
  DELETED_BLUETOOTH_READ = 345,
  WEBREQUESTINTERNAL_EVENTHANDLED = 346,
  EVENTS_ADDRULES = 347,
  CONTEXTMENUS_CREATE = 348,
  MEDIAPLAYERPRIVATE_GETPLAYLIST = 349,
  DOWNLOADS_ERASE = 350,
  DELETED_EXPERIMENTAL_RECORD_CAPTUREURLS = 351,
  TTS_ISSPEAKING = 352,
  BOOKMARKS_REMOVETREE = 353,
  FILEMANAGERPRIVATE_SEARCHDRIVE = 354,
  DELETED_EXPERIMENTAL_SYSTEMINFO_CPU_GET = 355,
  FILEMANAGERPRIVATE_SETPREFERENCES = 356,
  FONTSETTINGS_SETFONT = 357,
  SOCKET_GETNETWORKLIST = 358,
  BOOKMARKS_MOVE = 359,
  WALLPAPERPRIVATE_MINIMIZEINACTIVEWINDOWS = 360,
  STORAGE_REMOVE = 361,
  AUTOTESTPRIVATE_LOGINSTATUS = 362,
  TABS_CREATE = 363,
  FILEMANAGERPRIVATE_CANCELDIALOG = 364,
  BROWSINGDATA_REMOVECOOKIES = 365,
  FILESYSTEM_CHOOSEENTRY = 366,
  MEDIAPLAYERPRIVATE_PLAY = 367,
  WEBSTOREPRIVATE_GETWEBGLSTATUS = 368,
  SOCKET_SENDTO = 369,
  BROWSINGDATA_REMOVEFILESYSTEMS = 370,
  WALLPAPERPRIVATE_GETSTRINGS = 371,
  BROWSINGDATA_REMOVECACHE = 372,
  BOOKMARKMANAGERPRIVATE_RECORDLAUNCH = 373,
  BROWSERACTION_DISABLE = 374,
  DELETED_EXPERIMENTAL_SPEECHINPUT_ISRECORDING = 375,
  APP_WINDOW_CREATE = 376,
  RUNTIME_RELOAD = 377,
  EXPERIMENTAL_POWER_REQUESTKEEPAWAKE = 378,
  SYSTEMINDICATOR_SETICON = 379,
  FILEMANAGERPRIVATE_ADDMOUNT = 380,
  APP_CURRENTWINDOWINTERNAL_FOCUS = 381,
  EVENTS_REMOVERULES = 382,
  DOWNLOADS_DOWNLOAD = 383,
  WINDOWS_GETALL = 384,
  DELETED_FILEMANAGERPRIVATE_TOGGLEFULLSCREEN = 385,
  APP_CURRENTWINDOWINTERNAL_RESTORE = 386,
  DELETED_WEBSOCKETPROXYPRIVATE_GETPASSPORTFORTCP = 387,
  PAGEACTION_HIDE = 388,
  DELETED_EXPERIMENTAL_SYSTEMINFO_STORAGE_GET = 389,
  DOWNLOADS_ACCEPTDANGER = 390,
  DELETED_WEBSOCKETPROXYPRIVATE_GETURLFORTCP = 391,
  FILEMANAGERPRIVATE_GETSIZESTATS = 392,
  DOWNLOADS_RESUME = 393,
  COOKIES_GETALLCOOKIESTORES = 394,
  DELETED_MEDIAGALLERIESPRIVATE_ADDGALLERYWATCH = 395,
  DELETED_MEDIAGALLERIESPRIVATE_REMOVEGALLERYWATCH = 396,
  DELETED_WEBVIEW_EXECUTESCRIPT = 397,
  NOTIFICATIONS_UPDATE = 398,
  NOTIFICATIONS_CLEAR = 399,
  DELETED_SESSIONRESTORE_GETRECENTLYCLOSED = 400,
  DELETED_SESSIONRESTORE_RESTORE = 401,
  MANAGEMENT_UNINSTALLSELF = 402,
  ECHOPRIVATE_GETOOBETIMESTAMP = 403,
  FILEMANAGERPRIVATEINTERNAL_VALIDATEPATHNAMELENGTH = 404,
  BROWSINGDATA_SETTINGS = 405,
  WEBSTOREPRIVATE_GETISLAUNCHERENABLED = 406,
  NETWORKINGPRIVATE_GETPROPERTIES = 407,
  NETWORKINGPRIVATE_GETVISIBLENETWORKS = 408,
  NETWORKINGPRIVATE_STARTCONNECT = 409,
  NETWORKINGPRIVATE_STARTDISCONNECT = 410,
  DELETED_MEDIAGALLERIESPRIVATE_GETALLGALLERYWATCH = 411,
  DELETED_MEDIAGALLERIESPRIVATE_REMOVEALLGALLERYWATCH = 412,
  FILEMANAGERPRIVATE_SEARCHDRIVEMETADATA = 413,
  DELETED_ECHOPRIVATE_CHECKALLOWREDEEMOFFERS = 414,
  DELETED_MEDIAGALLERIESPRIVATE_EJECTDEVICE = 415,
  FILEMANAGERPRIVATE_LOGOUTUSERFORREAUTHENTICATION = 416,
  DEVELOPERPRIVATE_CHOOSEPATH = 417,
  DEVELOPERPRIVATE_PACKDIRECTORY = 418,
  NETWORKINGPRIVATE_VERIFYDESTINATION = 419,
  NETWORKINGPRIVATE_VERIFYANDENCRYPTCREDENTIALS = 420,
  NETWORKINGPRIVATE_VERIFYANDENCRYPTDATA = 421,
  DEVELOPERPRIVATE_RESTART = 422,
  DELETED_DEVELOPERPRIVATE_ALLOWINCOGNITO = 423,
  INPUT_IME_DELETESURROUNDINGTEXT = 424,
  DELETED_FILEMANAGERPRIVATE_OPENNEWWINDOW = 425,
  CLOUDPRINTPRIVATE_GETCLIENTID = 426,
  ECHOPRIVATE_GETUSERCONSENT = 427,
  SYNCFILESYSTEM_SETCONFLICTRESOLUTIONPOLICY = 428,
  SYNCFILESYSTEM_GETCONFLICTRESOLUTIONPOLICY = 429,
  NETWORKINGPRIVATE_SETPROPERTIES = 430,
  NETWORKINGPRIVATE_GETSTATE = 431,
  POWER_REQUESTKEEPAWAKE = 432,
  POWER_RELEASEKEEPAWAKE = 433,
  WALLPAPERPRIVATE_SETCUSTOMWALLPAPERLAYOUT = 434,
  DOWNLOADSINTERNAL_DETERMINEFILENAME = 435,
  SYNCFILESYSTEM_GETFILESYNCSTATUSES = 436,
  DELETED_MEDIAGALLERIESPRIVATE_GETHANDLERS = 437,
  WALLPAPERPRIVATE_RESETWALLPAPER = 438,
  DEVELOPERPRIVATE_PERMISSIONS = 439,
  WEBSTOREPRIVATE_ENABLEAPPLAUNCHER = 440,
  APP_CURRENTWINDOWINTERNAL_FULLSCREEN = 441,
  DEVELOPERPRIVATE_LOADUNPACKEDCROS = 442,
  NETWORKINGPRIVATE_REQUESTNETWORKSCAN = 443,
  ENTERPRISE_PLATFORMKEYSPRIVATE_CHALLENGEMACHINEKEY = 444,
  ENTERPRISE_PLATFORMKEYSPRIVATE_CHALLENGEUSERKEY = 445,
  DEVELOPERPRIVATE_LOADPROJECTTOSYNCFS = 446,
  DEVELOPERPRIVATE_GETPROJECTSINFO = 447,
  DEVELOPERPRIVATE_LOADPROJECT = 448,
  COMMANDLINEPRIVATE_HASSWITCH = 449,
  NETWORKINGPRIVATE_GETMANAGEDPROPERTIES = 450,
  DELETED_LOCATION_WATCHLOCATION = 451,
  DELETED_LOCATION_CLEARWATCH = 452,
  DELETED_BLUETOOTH_ADDPROFILE = 453,
  DELETED_BLUETOOTH_REMOVEPROFILE = 454,
  DELETED_BLUETOOTH_GETPROFILES = 455,
  EXPERIMENTAL_IDENTITY_REMOVECACHEDAUTHTOKEN = 456,
  AUDIO_GETINFO = 457,
  AUDIO_SETACTIVEDEVICES = 458,
  AUDIO_SETPROPERTIES = 459,
  USB_RESETDEVICE = 460,
  SOCKET_MULTICAST_JOIN_GROUP = 461,
  SOCKET_MULTICAST_LEAVE_GROUP = 462,
  SOCKET_MULTICAST_SET_TIME_TO_LIVE = 463,
  SOCKET_MULTICAST_SET_LOOPBACK_MODE = 464,
  SOCKET_MULTICAST_GET_JOINED_GROUPS = 465,
  ACCESSIBILITY_PRIVATE_SETNATIVEACCESSIBILITYENABLED = 466,
  RUNTIME_GETPLATFORMINFO = 467,
  DELETED_WEBVIEW_INSERTCSS = 468,
  METRICSPRIVATE_GETISCRASHRECORDINGENABLED = 469,
  DELETED_IDENTITYPRIVATE_GETSTRINGS = 470,
  NOTIFICATIONS_GET_ALL = 471,
  USB_LISTINTERFACES = 472,
  FILESYSTEM_RETAINENTRY = 473,
  FILESYSTEM_ISRESTORABLE = 474,
  FILESYSTEM_RESTOREENTRY = 475,
  RUNTIME_SETUNINSTALLURL = 476,
  INPUTMETHODPRIVATE_STARTIME = 477,
  MUSICMANAGERPRIVATE_GETDEVICEID = 478,
  TTS_PAUSE = 479,
  TTS_RESUME = 480,
  DELETED_SYSTEMINFO_CPU_GET = 481,
  DELETED_EXPERIMENTAL_SYSTEMINFO_STORAGE_ADDWATCH = 482,
  DELETED_EXPERIMENTAL_SYSTEMINFO_STORAGE_REMOVEWATCH = 483,
  DELETED_EXPERIMENTAL_SYSTEMINFO_STORAGE_GETALLWATCH = 484,
  DELETED_EXPERIMENTAL_SYSTEMINFO_STORAGE_REMOVEALLWATCH = 485,
  DELETED_SYSTEMINFO_MEMORY_GET = 486,
  ACTIVITYLOGPRIVATE_GETEXTENSIONACTIVITIES = 487,
  RUNTIME_GETPACKAGEDIRECTORYENTRY = 488,
  DELETED_SYSTEMINFO_DISPLAY_SETDISPLAYPROPERTIES = 489,
  FEEDBACKPRIVATE_GETUSEREMAIL = 490,
  FEEDBACKPRIVATE_GETSYSTEMINFORMATION = 491,
  FEEDBACKPRIVATE_SENDFEEDBACK = 492,
  VIRTUALKEYBOARDPRIVATE_INSERTTEXT = 493,
  DIAGNOSTICS_SENDPACKET = 494,
  METRICSPRIVATE_GETFIELDTRIAL = 495,
  FILEMANAGERPRIVATE_ZOOM = 496,
  DELETED_WEBVIEW_GO = 497,
  WEBSTOREPRIVATE_ISININCOGNITOMODEFUNCTION = 498,
  DELETED_FILEMANAGERPRIVATE_REQUESTACCESSTOKEN = 499,
  DELETED_WEBVIEW_STOP = 500,
  DELETED_WEBVIEW_RELOAD = 501,
  DELETED_WEBVIEW_TERMINATE = 502,
  DELETED_TYPES_PRIVATE_CHROMEDIRECTSETTING_GET = 503,
  DELETED_TYPES_PRIVATE_CHROMEDIRECTSETTING_SET = 504,
  DELETED_TYPES_PRIVATE_CHROMEDIRECTSETTING_CLEAR = 505,
  DELETED_EXPERIMENTAL_SYSTEMINFO_STORAGE_EJECTDEVICE = 506,
  SYSTEM_CPU_GETINFO = 507,
  BOOKMARKMANAGERPRIVATE_REMOVETREES = 508,
  SYSTEM_DISPLAY_GETINFO = 509,
  SYSTEM_DISPLAY_SETDISPLAYPROPERTIES = 510,
  SYSTEM_MEMORY_GETINFO = 511,
  FILEMANAGERPRIVATEINTERNAL_GETSHAREURL = 512,
  SYSTEM_STORAGE_GETINFO = 513,
  SYSTEM_STORAGE_EJECTDEVICE = 514,
  DELETED_SYSTEM_STORAGE_ADDAVAILABLECAPACITYWATCH = 515,
  DELETED_SYSTEM_STORAGE_REMOVEAVAILABLECAPACITYWATCH = 516,
  DELETED_SYSTEM_STORAGE_GETALLAVAILABLECAPACITYWATCHES = 517,
  DELETED_SYSTEM_STORAGE_REMOVEALLAVAILABLECAPACITYWATCHES = 518,
  DOWNLOADS_REMOVEFILE = 519,
  DOWNLOADS_SHOWDEFAULTFOLDER = 520,
  DELETED_INFOBARS_SHOW = 521,
  DOWNLOADS_SETSHELFENABLED = 522,
  IMAGEWRITER_WRITEFROMURL = 523,
  IMAGEWRITER_WRITEFROMFILE = 524,
  IMAGEWRITER_CANCELWRITE = 525,
  IMAGEWRITER_DESTROYPARTITIONS = 526,
  FEEDBACKPRIVATE_GETSTRINGS = 527,
  DELETED_LOGPRIVATE_GETHISTORICAL = 528,
  VIRTUALKEYBOARDPRIVATE_MOVECURSOR = 529,
  METRICSPRIVATE_GETVARIATIONPARAMS = 530,
  DELETED_WEBVIEW_SETPERMISSION = 531,
  DESKTOPCAPTURE_CHOOSEDESKTOPMEDIA = 532,
  APP_CURRENTWINDOWINTERNAL_SETSHAPE = 533,
  PROCESSES_GETPROCESSINFO = 534,
  PROCESSES_GETPROCESSIDFORTAB = 535,
  PROCESSES_TERMINATE = 536,
  SOCKETS_UDP_CREATE = 537,
  SOCKETS_UDP_UPDATE = 538,
  SOCKETS_UDP_BIND = 539,
  SOCKETS_UDP_SEND = 540,
  SOCKETS_UDP_CLOSE = 541,
  SOCKETS_UDP_GETINFO = 542,
  SOCKETS_UDP_GETSOCKETS = 543,
  SOCKETS_UDP_JOINGROUP = 544,
  SOCKETS_UDP_LEAVEGROUP = 545,
  SOCKETS_UDP_SETMULTICASTTIMETOLIVE = 546,
  SOCKETS_UDP_SETMULTICASTLOOPBACKMODE = 547,
  SOCKETS_UDP_GETJOINEDGROUPS = 548,
  SIGNED_IN_DEVICES_GET = 549,
  AUTOTESTPRIVATE_SIMULATEASANMEMORYBUG = 550,
  DELETED_WEBVIEW_CLEARDATA = 551,
  SESSIONS_GETRECENTLYCLOSED = 552,
  SESSIONS_GETDEVICES = 553,
  SESSIONS_RESTORE = 554,
  SYNCFILESYSTEM_GETSERVICESTATUS = 555,
  ECHOPRIVATE_SETOFFERINFO = 556,
  ECHOPRIVATE_GETOFFERINFO = 557,
  DEVELOPERPRIVATE_ISPROFILEMANAGED = 558,
  DELETED_FILEMANAGERPRIVATE_INSTALLWEBSTOREITEM = 559,
  FILEMANAGERPRIVATEINTERNAL_STARTCOPY = 560,
  FILEMANAGERPRIVATE_CANCELCOPY = 561,
  NETWORKINGPRIVATE_CREATENETWORK = 562,
  BRAILLEDISPLAYPRIVATE_GETDISPLAYSTATE = 563,
  BRAILLEDISPLAYPRIVATE_WRITEDOTS = 564,
  USB_GETDEVICES = 565,
  USB_REQUESTACCESS = 566,
  USB_OPENDEVICE = 567,
  ACTIVITYLOGPRIVATE_DELETEDATABASE = 568,
  ACTIVITYLOGPRIVATE_DELETEURLS = 569,
  FILEMANAGERPRIVATE_REQUESTWEBSTOREACCESSTOKEN = 570,
  IMAGEWRITER_LISTREMOVABLESTORAGEDEVICES = 571,
  WALLPAPER_SETWALLPAPER = 572,
  VIRTUALKEYBOARDPRIVATE_HIDEKEYBOARD = 573,
  AUTOTESTPRIVATE_LOCKSCREEN = 574,
  WEBRTCLOGGINGPRIVATE_SETMETADATA = 575,
  WEBRTCLOGGINGPRIVATE_START = 576,
  WEBRTCLOGGINGPRIVATE_SETUPLOADONRENDERCLOSE = 577,
  WEBRTCLOGGINGPRIVATE_STOP = 578,
  WEBRTCLOGGINGPRIVATE_UPLOAD = 579,
  WEBRTCLOGGINGPRIVATE_DISCARD = 580,
  DELETED_WEBVIEW_OVERRIDEUSERAGENT = 581,
  DELETED_PRINCIPALSPRIVATE_SHOWAVATARBUBBLE = 582,
  DELETED_PRINCIPALSPRIVATE_SIGNOUT = 583,
  CAST_CHANNEL_OPEN = 584,
  CAST_CHANNEL_SEND = 585,
  CAST_CHANNEL_CLOSE = 586,
  RUNTIME_RESTART = 587,
  DESKTOPCAPTURE_CANCELCHOOSEDESKTOPMEDIA = 588,
  APP_CURRENTWINDOWINTERNAL_SETALWAYSONTOP = 589,
  SOCKETS_TCP_CREATE = 590,
  SOCKETS_TCP_UPDATE = 591,
  SOCKETS_TCP_SETPAUSED = 592,
  SOCKETS_TCP_SETKEEPALIVE = 593,
  SOCKETS_TCP_SETNODELAY = 594,
  SOCKETS_TCP_CONNECT = 595,
  SOCKETS_TCP_DISCONNECT = 596,
  SOCKETS_TCP_SEND = 597,
  SOCKETS_TCP_CLOSE = 598,
  SOCKETS_TCP_GETINFO = 599,
  SOCKETS_TCP_GETSOCKETS = 600,
  NETWORKINGPRIVATE_GETENABLEDNETWORKTYPES = 601,
  NETWORKINGPRIVATE_ENABLENETWORKTYPE = 602,
  NETWORKINGPRIVATE_DISABLENETWORKTYPE = 603,
  SOCKETS_TCP_SERVER_CREATE = 604,
  SOCKETS_TCP_SERVER_UPDATE = 605,
  SOCKETS_TCP_SERVER_SETPAUSED = 606,
  SOCKETS_TCP_SERVER_LISTEN = 607,
  SOCKETS_TCP_SERVER_DISCONNECT = 608,
  SOCKETS_TCP_SERVER_CLOSE = 609,
  SOCKETS_TCP_SERVER_GETINFO = 610,
  SOCKETS_TCP_SERVER_GETSOCKETS = 611,
  SYSTEM_STORAGE_GETAVAILABLECAPACITY = 612,
  BROWSERACTION_OPEN_POPUP = 613,
  WEBRTC_AUDIO_PRIVATE_GET_SINKS = 614,
  DELETED_WEBRTC_AUDIO_PRIVATE_GET_ACTIVE_SINK = 615,
  DELETED_WEBRTC_AUDIO_PRIVATE_SET_ACTIVE_SINK = 616,
  WEBRTC_AUDIO_PRIVATE_GET_ASSOCIATED_SINK = 617,
  VIRTUALKEYBOARDPRIVATE_KEYBOARDLOADED = 618,
  DELETED_APP_CURRENTWINDOWINTERNAL_SETMINWIDTH = 619,
  DELETED_APP_CURRENTWINDOWINTERNAL_SETMINHEIGHT = 620,
  DELETED_APP_CURRENTWINDOWINTERNAL_SETMAXWIDTH = 621,
  DELETED_APP_CURRENTWINDOWINTERNAL_SETMAXHEIGHT = 622,
  SYSTEMPRIVATE_GETAPIKEY = 623,
  CHROMEOSINFOPRIVATE_SET = 624,
  BOOKMARKMANAGERPRIVATE_GETMETAINFO = 625,
  BOOKMARKMANAGERPRIVATE_SETMETAINFO = 626,
  FILESYSTEMPROVIDER_MOUNT = 627,
  AUTOTESTPRIVATE_GETEXTENSIONSINFO = 628,
  DELETED_SCREENLOCKPRIVATE_GETLOCKED = 629,
  DELETED_SCREENLOCKPRIVATE_SETLOCKED = 630,
  DELETED_SCREENLOCKPRIVATE_SHOWMESSAGE = 631,
  FEEDBACKPRIVATE_GETHISTOGRAMS = 632,
  SYSTEM_NETWORK_GETNETWORKINTERFACES = 633,
  SERIAL_GETDEVICES = 634,
  SERIAL_UPDATE = 635,
  SERIAL_SETPAUSED = 636,
  SERIAL_GETINFO = 637,
  SERIAL_GETCONNECTIONS = 638,
  SERIAL_SEND = 639,
  GCM_REGISTER = 640,
  GCM_SEND = 641,
  SERIAL_CONNECT = 642,
  SERIAL_DISCONNECT = 643,
  DELETED_MEDIAGALLERIES_GETALLMEDIAFILESYSTEMMETADATA = 644,
  FIRSTRUNPRIVATE_GETLOCALIZEDSTRINGS = 645,
  FIRSTRUNPRIVATE_LAUNCHTUTORIAL = 646,
  SOCKETS_UDP_SETPAUSED = 647,
  DELETED_WEBVIEW_CAPTUREVISIBLEREGION = 648,
  MEDIAGALLERIES_GETMETADATA = 649,
  INPUT_IME_SENDKEYEVENTS = 650,
  VIRTUALKEYBOARDPRIVATE_LOCKKEYBOARD = 651,
  DELETED_SCREENLOCKPRIVATE_SHOWCUSTOMICON = 652,
  INPUT_IME_HIDEINPUTVIEW = 653,
  BOOKMARKMANAGERPRIVATE_UNDO = 654,
  BOOKMARKMANAGERPRIVATE_REDO = 655,
  BOOKMARKMANAGERPRIVATE_UNDOINFO = 656,
  BOOKMARKMANAGERPRIVATE_REDOINFO = 657,
  MEDIAGALLERIES_ADDUSERSELECTEDFOLDER = 658,
  DELETED_PREFERENCESPRIVATE_GETSYNCCATEGORIESWITHOUTPASSPHRASE = 659,
  DELETED_READINGLISTPRIVATE_ADDENTRY = 660,
  DELETED_READINGLISTPRIVATE_REMOVEENTRY = 661,
  DELETED_READINGLISTPRIVATE_GETENTRIES = 662,
  DELETED_MEDIAGALLERIES_STARTMEDIASCAN = 663,
  DELETED_MEDIAGALLERIES_CANCELMEDIASCAN = 664,
  DELETED_MEDIAGALLERIES_ADDSCANRESULTS = 665,
  DELETED_LOGPRIVATE_STARTNETINTERNALSWATCH = 666,
  DELETED_LOGPRIVATE_STOPNETINTERNALSWATCH = 667,
  FILEMANAGERPRIVATE_GETPROFILES = 668,
  DELETED_FILEMANAGERPRIVATE_VISITDESKTOP = 669,
  VIRTUALKEYBOARDPRIVATE_GETKEYBOARDCONFIG = 670,
  HID_GETDEVICES = 671,
  HID_CONNECT = 672,
  HID_DISCONNECT = 673,
  HID_RECEIVE = 674,
  HID_SEND = 675,
  HID_RECEIVEFEATUREREPORT = 676,
  HID_SENDFEATUREREPORT = 677,
  HOTWORDPRIVATE_SETENABLED = 678,
  HOTWORDPRIVATE_GETSTATUS = 679,
  DELETED_APP_CURRENTWINDOWINTERNAL_SETBADGEICON = 680,
  DELETED_APP_CURRENTWINDOWINTERNAL_CLEARBADGE = 681,
  DELETED_WEBVIEW_SETZOOM = 682,
  DELETED_WEBVIEW_GETZOOM = 683,
  DEVELOPERPRIVATE_REQUESTFILESOURCE = 684,
  DEVELOPERPRIVATE_OPENDEVTOOLS = 685,
  ACTIVITYLOGPRIVATE_DELETEACTIVITIES = 686,
  NETWORKINGPRIVATE_SETWIFITDLSENABLEDSTATE = 687,
  NETWORKINGPRIVATE_GETWIFITDLSSTATUS = 688,
  DELETED_SCREENLOCKPRIVATE_HIDECUSTOMICON = 689,
  DELETED_SCREENLOCKPRIVATE_SETAUTHTYPE = 690,
  DELETED_SCREENLOCKPRIVATE_GETAUTHTYPE = 691,
  DELETED_SCREENLOCKPRIVATE_ACCEPTAUTHATTEMPT = 692,
  DELETED_WEBVIEW_FIND = 693,
  DELETED_WEBVIEW_STOPFINDING = 694,
  DELETED_WEBVIEW_CONTEXTMENUSCREATE = 695,
  DELETED_WEBVIEW_CONTEXTMENUSUPDATE = 696,
  DELETED_WEBVIEW_CONTEXTMENUSREMOVE = 697,
  DELETED_WEBVIEW_CONTEXTMENUSREMOVEALL = 698,
  AUTOMATIONINTERNAL_ENABLETAB = 699,
  APP_CURRENTWINDOWINTERNAL_SETSIZECONSTRAINTS = 700,
  BLUETOOTH_GETDEVICE = 701,
  GCM_UNREGISTER = 702,
  DELETED_FILEMANAGERPRIVATEINTERNAL_REQUESTDRIVESHARE = 703,
  METRICSPRIVATE_RECORDSPARSEVALUE = 704,
  HOTWORDPRIVATE_SETAUDIOLOGGINGENABLED = 705,
  BLUETOOTHPRIVATE_SETADAPTERSTATE = 706,
  DELETED_BLUETOOTHPRIVATE_ENABLEPAIRING = 707,
  DELETED_BLUETOOTHPRIVATE_DISABLEPAIRING = 708,
  BLUETOOTHPRIVATE_SETPAIRINGRESPONSE = 709,
  NETWORKINGPRIVATE_GETCAPTIVEPORTALSTATUS = 710,
  AUTOMATIONINTERNAL_PERFORMACTION = 711,
  DELETED_BLUETOOTH_UPDATE_SOCKET = 712,
  DELETED_BLUETOOTH_SET_SOCKET_PAUSED = 713,
  DELETED_BLUETOOTH_GET_SOCKET = 714,
  DELETED_BLUETOOTH_GET_SOCKETS = 715,
  FILESYSTEMPROVIDER_UNMOUNT = 716,
  FILESYSTEMPROVIDERINTERNAL_UNMOUNTREQUESTEDSUCCESS = 717,
  DELETED_FILESYSTEMPROVIDERINTERNAL_UNMOUNTREQUESTEDERROR = 718,
  DELETED_MEDIAGALLERIES_DROPPERMISSIONFORMEDIAFILESYSTEM = 719,
  WEBCAMPRIVATE_SET = 720,
  WEBCAMPRIVATE_RESET = 721,
  WEBCAMPRIVATE_GET = 722,
  BLUETOOTHLOWENERGY_GETSERVICE = 723,
  BLUETOOTHLOWENERGY_GETSERVICES = 724,
  BLUETOOTHLOWENERGY_GETCHARACTERISTIC = 725,
  BLUETOOTHLOWENERGY_GETCHARACTERISTICS = 726,
  BLUETOOTHLOWENERGY_GETINCLUDEDSERVICES = 727,
  BLUETOOTHLOWENERGY_GETDESCRIPTOR = 728,
  BLUETOOTHLOWENERGY_GETDESCRIPTORS = 729,
  BLUETOOTHLOWENERGY_READCHARACTERISTICVALUE = 730,
  BLUETOOTHLOWENERGY_WRITECHARACTERISTICVALUE = 731,
  BLUETOOTHLOWENERGY_READDESCRIPTORVALUE = 732,
  BLUETOOTHLOWENERGY_WRITEDESCRIPTORVALUE = 733,
  BOOKMARKMANAGERPRIVATE_CREATEWITHMETAINFO = 734,
  BOOKMARKMANAGERPRIVATE_UPDATEMETAINFO = 735,
  BLUETOOTHSOCKET_CREATE = 736,
  BLUETOOTHSOCKET_UPDATE = 737,
  BLUETOOTHSOCKET_SETPAUSED = 738,
  BLUETOOTHSOCKET_LISTENUSINGRFCOMM = 739,
  BLUETOOTHSOCKET_LISTENUSINGINSECURERFCOMM = 740,
  BLUETOOTHSOCKET_LISTENUSINGL2CAP = 741,
  BLUETOOTHSOCKET_CONNECT = 742,
  BLUETOOTHSOCKET_DISCONNECT = 743,
  BLUETOOTHSOCKET_CLOSE = 744,
  BLUETOOTHSOCKET_SEND = 745,
  BLUETOOTHSOCKET_GETINFO = 746,
  BLUETOOTHSOCKET_GETSOCKETS = 747,
  DELETED_WEBSTOREPRIVATE_SIGNINFUNCTION = 748,
  DELETED_SHELL_CREATEWINDOW = 749,
  FILESYSTEMPROVIDERINTERNAL_GETMETADATAREQUESTEDSUCCESS = 750,
  DELETED_FILESYSTEMPROVIDERINTERNAL_GETMETADATAREQUESTEDERROR = 751,
  BROWSER_OPENTAB = 752,
  MANAGEMENT_CREATEAPPSHORTCUT = 753,
  DELETED_WEBVIEW_SHOWCONTEXTMENU = 754,
  WEBRTCLOGGINGPRIVATE_STARTRTPDUMP = 755,
  WEBRTCLOGGINGPRIVATE_STOPRTPDUMP = 756,
  AUTOMATIONINTERNAL_ENABLEDESKTOP = 757,
  HOTWORDPRIVATE_SETHOTWORDSESSIONSTATE = 758,
  HOTWORDPRIVATE_NOTIFYHOTWORDRECOGNITION = 759,
  FILESYSTEMPROVIDERINTERNAL_READDIRECTORYREQUESTEDSUCCESS = 760,
  DELETED_FILESYSTEMPROVIDERINTERNAL_READDIRECTORYREQUESTEDERROR = 761,
  DELETED_LEDGER_BATCHEXECUTE = 762,
  DELETED_FILESYSTEMPROVIDERINTERNAL_OPENFILEREQUESTEDSUCCESS = 763,
  DELETED_FILESYSTEMPROVIDERINTERNAL_OPENFILEREQUESTEDERROR = 764,
  DELETED_FILESYSTEMPROVIDERINTERNAL_CLOSEFILEREQUESTEDSUCCESS = 765,
  DELETED_FILESYSTEMPROVIDERINTERNAL_CLOSEFILEREQUESTEDERROR = 766,
  DELETED_SYNCEDNOTIFICATIONSPRIVATE_GETINITIALDATA = 767,
  DELETED_SYNCEDNOTIFICATIONSPRIVATE_UPDATENOTIFICATION = 768,
  DELETED_SYNCEDNOTIFICATIONSPRIVATE_SETRENDERCONTEXT = 769,
  IDENTITY_GETACCOUNTS = 770,
  FILEMANAGERPRIVATE_RESOLVEISOLATEDENTRIES = 771,
  FILESYSTEMPROVIDERINTERNAL_READFILEREQUESTEDSUCCESS = 772,
  DELETED_FILESYSTEMPROVIDERINTERNAL_READFILEREQUESTEDERROR = 773,
  NETWORKINGPRIVATE_GETNETWORKS = 774,
  DELETED_WEBVIEW_SETNAME = 775,
  ENTERPRISE_PLATFORMKEYSINTERNAL_GENERATEKEY = 776,
  PLATFORMKEYSINTERNAL_SIGN = 777,
  ENTERPRISE_PLATFORMKEYSINTERNAL_GETTOKENS = 778,
  ENTERPRISE_PLATFORMKEYS_GETCERTIFICATES = 779,
  ENTERPRISE_PLATFORMKEYS_IMPORTCERTIFICATE = 780,
  ENTERPRISE_PLATFORMKEYS_REMOVECERTIFICATE = 781,
  FILEMANAGERPRIVATE_OPENINSPECTOR = 782,
  DELETED_STREAMSPRIVATE_ABORT = 783,
  MANAGEMENT_SETLAUNCHTYPE = 784,
  MANAGEMENT_GENERATEAPPFORLINK = 785,
  DELETED_GUESTVIEWINTERNAL_ALLOCATEINSTANCEID = 786,
  DELETED_WEBVIEW_NAVIGATE = 787,
  INPUTMETHODPRIVATE_GETCURRENTINPUTMETHOD = 788,
  INPUTMETHODPRIVATE_SETCURRENTINPUTMETHOD = 789,
  INPUTMETHODPRIVATE_GETINPUTMETHODS = 790,
  IDENTITY_GETPROFILEUSERINFO = 791,
  VIRTUALKEYBOARDPRIVATE_OPENSETTINGS = 792,
  BLUETOOTHLOWENERGY_CONNECT = 793,
  BLUETOOTHLOWENERGY_DISCONNECT = 794,
  WEBSTOREPRIVATE_GETEPHEMERALAPPSENABLED = 795,
  WEBSTOREPRIVATE_LAUNCHEPHEMERALAPP = 796,
  FILESYSTEMPROVIDERINTERNAL_OPERATIONREQUESTEDERROR = 797,
  FILESYSTEMPROVIDERINTERNAL_OPERATIONREQUESTEDSUCCESS = 798,
  GUESTVIEWINTERNAL_CREATEGUEST = 799,
  WEBVIEWINTERNAL_CAPTUREVISIBLEREGION = 800,
  WEBVIEWINTERNAL_CONTEXTMENUSCREATE = 801,
  WEBVIEWINTERNAL_CONTEXTMENUSREMOVE = 802,
  WEBVIEWINTERNAL_CONTEXTMENUSREMOVEALL = 803,
  WEBVIEWINTERNAL_CONTEXTMENUSUPDATE = 804,
  WEBVIEWINTERNAL_CLEARDATA = 805,
  WEBVIEWINTERNAL_EXECUTESCRIPT = 806,
  WEBVIEWINTERNAL_FIND = 807,
  WEBVIEWINTERNAL_GETZOOM = 808,
  WEBVIEWINTERNAL_GO = 809,
  WEBVIEWINTERNAL_INSERTCSS = 810,
  WEBVIEWINTERNAL_NAVIGATE = 811,
  WEBVIEWINTERNAL_OVERRIDEUSERAGENT = 812,
  WEBVIEWINTERNAL_RELOAD = 813,
  WEBVIEWINTERNAL_SETNAME = 814,
  WEBVIEWINTERNAL_SETPERMISSION = 815,
  WEBVIEWINTERNAL_SETZOOM = 816,
  WEBVIEWINTERNAL_SHOWCONTEXTMENU = 817,
  WEBVIEWINTERNAL_STOP = 818,
  WEBVIEWINTERNAL_STOPFINDING = 819,
  WEBVIEWINTERNAL_TERMINATE = 820,
  BLUETOOTHLOWENERGY_STARTCHARACTERISTICNOTIFICATIONS = 821,
  BLUETOOTHLOWENERGY_STOPCHARACTERISTICNOTIFICATIONS = 822,
  DELETED_GCDPRIVATE_GETCLOUDDEVICELIST = 823,
  DELETED_GCDPRIVATE_QUERYFORNEWLOCALDEVICES = 824,
  DELETED_GCDPRIVATE_PREFETCHWIFIPASSWORD = 825,
  DELETED_GCDPRIVATE_ESTABLISHSESSION = 826,
  DELETED_GCDPRIVATE_SENDMESSAGE = 827,
  DELETED_GCDPRIVATE_CONFIRMCODE = 828,
  DELETED_GCDPRIVATE_TERMINATESESSION = 829,
  TABS_SETZOOM = 830,
  TABS_GETZOOM = 831,
  TABS_SETZOOMSETTINGS = 832,
  TABS_GETZOOMSETTINGS = 833,
  DELETED_GCDPRIVATE_GETCOMMANDDEFINITIONS = 834,
  DELETED_GCDPRIVATE_INSERTCOMMAND = 835,
  DELETED_GCDPRIVATE_GETCOMMAND = 836,
  DELETED_GCDPRIVATE_CANCELCOMMAND = 837,
  DELETED_GCDPRIVATE_GETCOMMANDSLIST = 838,
  APPVIEWINTERNAL_ATTACHFRAME = 839,
  APPVIEWINTERNAL_DENYREQUEST = 840,
  FILEMANAGERPRIVATEINTERNAL_GETDOWNLOADURL = 841,
  DELETED_EASYUNLOCKPRIVATE_PERFORMECDHKEYAGREEMENT = 842,
  DELETED_EASYUNLOCKPRIVATE_GENERATEECP256KEYPAIR = 843,
  DELETED_EASYUNLOCKPRIVATE_CREATESECUREMESSAGE = 844,
  DELETED_EASYUNLOCKPRIVATE_UNWRAPSECUREMESSAGE = 845,
  DELETED_LOGPRIVATE_STARTEVENTRECODER = 846,
  DELETED_LOGPRIVATE_STOPEVENTRECODER = 847,
  DELETED_LOGPRIVATE_DUMPLOGS = 848,
  SOCKET_SECURE = 849,
  SOCKETS_TCP_SECURE = 850,
  DELETED_EXPERIENCESAMPLINGPRIVATE_GETBROWSERINFO = 851,
  DELETED_EASYUNLOCKPRIVATE_SEEKBLUETOOTHDEVICEBYADDRESS = 852,
  DELETED_EASYUNLOCKPRIVATE_GETSTRINGS = 853,
  DELETED_NOTIFICATIONPROVIDER_SENDONCLEAR = 854,
  DELETED_NOTIFICATIONPROVIDER_NOTIFYONCLEARED = 855,
  DELETED_NOTIFICATIONPROVIDER_NOTIFYONCLICKED = 856,
  DELETED_NOTIFICATIONPROVIDER_NOTIFYONBUTTONCLICKED = 857,
  DELETED_NOTIFICATIONPROVIDER_NOTIFYONPERMISSIONLEVELCHANGED = 858,
  DELETED_NOTIFICATIONPROVIDER_NOTIFYONSHOWSETTINGS = 859,
  DELETED_NOTIFICATIONPROVIDER_GETNOTIFIER = 860,
  DELETED_NOTIFICATIONPROVIDER_GETALLNOTIFIERS = 861,
  DELETED_GCDPRIVATE_GETPREFETCHEDWIFINAMELIST = 862,
  GUESTVIEWINTERNAL_SETAUTOSIZE = 863,
  DELETED_COPRESENCEPRIVATE_SENDFOUND = 864,
  DELETED_COPRESENCEPRIVATE_SENDSAMPLES = 865,
  DELETED_COPRESENCEPRIVATE_SENDDETECT = 866,
  DELETED_COPRESENCEPRIVATE_SENDINITIALIZED = 867,
  DELETED_COPRESENCE_EXECUTE = 868,
  DELETED_COPRESENCE_SETAPIKEY = 869,
  DELETED_FILESYSTEM_OBSERVEDIRECTORY = 870,
  DELETED_FILESYSTEM_UNOBSERVEENTRY = 871,
  DELETED_FILESYSTEM_GETOBSERVEDENTRIES = 872,
  BROWSINGDATA_REMOVESERVICEWORKERS = 873,
  DELETED_USBPRIVATE_GETDEVICES = 874,
  DELETED_USBPRIVATE_GETDEVICEINFO = 875,
  DELETED_EASYUNLOCKPRIVATE_UPDATESCREENLOCKSTATE = 876,
  CAST_CHANNEL_GETLOGS = 877,
  DELETED_EASYUNLOCKPRIVATE_SETPERMITACCESS = 878,
  DELETED_EASYUNLOCKPRIVATE_GETPERMITACCESS = 879,
  DELETED_EASYUNLOCKPRIVATE_CLEARPERMITACCESS = 880,
  DELETED_EASYUNLOCKPRIVATE_SETREMOTEDEVICES = 881,
  DELETED_EASYUNLOCKPRIVATE_GETREMOTEDEVICES = 882,
  FILESYSTEMPROVIDER_GETALL = 883,
  DELETED_EASYUNLOCKPRIVATE_CONNECTTOBLUETOOTHSERVICEINSECURELY = 884,
  MEDIAGALLERIES_ADDGALLERYWATCH = 885,
  MEDIAGALLERIES_REMOVEGALLERYWATCH = 886,
  DELETED_MEDIAGALLERIES_GETALLGALLERYWATCH = 887,
  DELETED_MEDIAGALLERIES_REMOVEALLGALLERYWATCH = 888,
  MANAGEMENT_GETSELF = 889,
  APP_CURRENTWINDOWINTERNAL_SETVISIBLEONALLWORKSPACES = 890,
  DELETED_EASYUNLOCKPRIVATE_GETSIGNINCHALLENGE = 891,
  DELETED_EASYUNLOCKPRIVATE_TRYSIGNINSECRET = 892,
  ACCESSIBILITY_PRIVATE_SETFOCUSRING = 893,
  USB_GETCONFIGURATION = 894,
  WEBVIEWINTERNAL_SETALLOWTRANSPARENCY = 895,
  DELETED_EASYUNLOCKPRIVATE_GETUSERINFO = 896,
  HOTWORDPRIVATE_GETLAUNCHSTATE = 897,
  HOTWORDPRIVATE_SETHOTWORDALWAYSONSEARCHENABLED = 898,
  WEBVIEWINTERNAL_LOADDATAWITHBASEURL = 899,
  GUESTVIEWINTERNAL_DESTROYGUEST = 900,
  DELETED_COPRESENCESOCKET_CREATEPEER = 901,
  DELETED_COPRESENCESOCKET_DESTROYPEER = 902,
  DELETED_COPRESENCESOCKET_SEND = 903,
  DELETED_COPRESENCESOCKET_DISCONNECT = 904,
  DELETED_BOOKMARKMANAGERPRIVATE_SETVERSION = 905,
  FILESYSTEMPROVIDER_NOTIFY = 906,
  USB_GETUSERSELECTEDDEVICES = 907,
  INPUTMETHODPRIVATE_GETINPUTMETHODCONFIG = 908,
  WALLPAPERPRIVATE_GETSYNCSETTING = 909,
  DELETED_COPRESENCE_SETAUTHTOKEN = 910,
  CAST_CHANNEL_SETAUTHORITYKEYS = 911,
  HOTWORDPRIVATE_STARTTRAINING = 912,
  HOTWORDPRIVATE_FINALIZESPEAKERMODEL = 913,
  HOTWORDPRIVATE_STOPTRAINING = 914,
  DELETED_GCDPRIVATE_STARTPAIRING = 915,
  VPNPROVIDER_CREATECONFIG = 916,
  VPNPROVIDER_DESTROYCONFIG = 917,
  VPNPROVIDER_SETPARAMETERS = 918,
  VPNPROVIDER_SENDPACKET = 919,
  VPNPROVIDER_NOTIFYCONNECTIONSTATECHANGED = 920,
  DOCUMENT_SCAN_SCAN = 921,
  DELETED_EASYUNLOCKPRIVATE_GETUSERIMAGE = 922,
  HOTWORDPRIVATE_NOTIFYSPEAKERMODELSAVED = 923,
  HOTWORDPRIVATE_GETLOCALIZEDSTRINGS = 924,
  HOTWORDPRIVATE_SETAUDIOHISTORYENABLED = 925,
  HOTWORDPRIVATE_GETAUDIOHISTORYENABLED = 926,
  DELETED_APP_CURRENTWINDOWINTERNAL_SETINTERCEPTALLKEYS = 927,
  DELETED_LAUNCHERPAGE_PUSHSUBPAGE = 928,
  DELETED_LAUNCHERPAGE_SHOW = 929,
  AUTOTESTPRIVATE_SETTOUCHPADSENSITIVITY = 930,
  AUTOTESTPRIVATE_SETTAPTOCLICK = 931,
  AUTOTESTPRIVATE_SETTHREEFINGERCLICK = 932,
  AUTOTESTPRIVATE_SETTAPDRAGGING = 933,
  AUTOTESTPRIVATE_SETNATURALSCROLL = 934,
  AUTOTESTPRIVATE_SETMOUSESENSITIVITY = 935,
  AUTOTESTPRIVATE_SETPRIMARYBUTTONRIGHT = 936,
  DELETED_COPRESENCEENDPOINTS_CREATELOCALENDPOINT = 937,
  DELETED_COPRESENCEENDPOINTS_DESTROYLOCALENDPOINT = 938,
  DELETED_COPRESENCEENDPOINTS_SEND = 939,
  DELETED_INLINE_INSTALL_PRIVATE_INSTALL = 940,
  DELETED_LAUNCHERPAGE_SETENABLED = 941,
  DELETED_CRYPTOTOKENPRIVATE_REQUESTPERMISSION = 942,
  BLUETOOTHPRIVATE_DISCONNECTALL = 943,
  FILEMANAGERPRIVATEINTERNAL_GETMIMETYPE = 944,
  USB_SETCONFIGURATION = 945,
  DELETED_EASYUNLOCKPRIVATE_GETCONNECTIONINFO = 946,
  FILEMANAGERPRIVATEINTERNAL_COMPUTECHECKSUM = 947,
  PRINTERPROVIDERINTERNAL_REPORTPRINTRESULT = 948,
  PRINTERPROVIDERINTERNAL_REPORTPRINTERCAPABILITY = 949,
  PRINTERPROVIDERINTERNAL_REPORTPRINTERS = 950,
  FILESYSTEMPROVIDER_GET = 951,
  FILEMANAGERPRIVATE_SEARCHFILESBYHASHES = 952,
  DELETED_EASYUNLOCKPRIVATE_SHOWERRORBUBBLE = 953,
  DELETED_EXTENSIONVIEWINTERNAL_NAVIGATE = 954,  // Obsolete: crbug.com/982858
  NETWORKING_CONFIG_SETNETWORKFILTER = 955,
  NETWORKING_CONFIG_FINISHAUTHENTICATION = 956,
  PLATFORMKEYSINTERNAL_SELECTCLIENTCERTIFICATES = 957,
  CRYPTOTOKENPRIVATE_CANORIGINASSERTAPPID = 958,
  DELETED_EASYUNLOCKPRIVATE_SETAUTOPAIRINGRESULT = 959,
  DELETED_FILEMANAGERPRIVATE_ISUMAENABLED = 960,
  WEBVIEWINTERNAL_SETALLOWSCALING = 961,
  PLATFORMKEYSINTERNAL_GETPUBLICKEY = 962,
  RUNTIME_OPENOPTIONSPAGE = 963,
  DELETED_AUDIOMODEM_TRANSMIT = 964,
  DELETED_AUDIOMODEM_STOPTRANSMIT = 965,
  DELETED_AUDIOMODEM_RECEIVE = 966,
  DELETED_AUDIOMODEM_STOPRECEIVE = 967,
  WEBRTCLOGGINGPRIVATE_STORE = 968,
  WEBRTCLOGGINGPRIVATE_UPLOADSTORED = 969,
  FILEMANAGERPRIVATEINTERNAL_SETENTRYTAG = 970,
  INPUTMETHODPRIVATE_FETCHALLDICTIONARYWORDS = 971,
  INPUTMETHODPRIVATE_ADDWORDTODICTIONARY = 972,
  DASHBOARDPRIVATE_SHOWPERMISSIONPROMPTFORDELEGATEDINSTALL = 973,
  PRINTERPROVIDERINTERNAL_GETPRINTDATA = 974,
  VIRTUALKEYBOARDPRIVATE_SETMODE = 975,
  NETWORKINGPRIVATE_STARTACTIVATE = 976,
  FILEMANAGERPRIVATE_GRANTACCESS = 977,
  DEVELOPERPRIVATE_GETEXTENSIONSINFO = 978,
  DEVELOPERPRIVATE_GETEXTENSIONINFO = 979,
  FILEMANAGERPRIVATE_ENABLEEXTERNALFILESCHEME = 980,
  DEVELOPERPRIVATE_UPDATEEXTENSIONCONFIGURATION = 981,
  DEVELOPERPRIVATE_DELETEEXTENSIONERRORS = 982,
  FILEMANAGERPRIVATE_ISPIEXLOADERENABLED = 983,
  HOTWORDPRIVATE_SPEAKERMODELEXISTSRESULT = 984,
  SETTINGSPRIVATE_SETPREF = 985,
  DELETED_SETTINGSPRIVATE_SETDOUBLEPREF = 986,
  DELETED_SETTINGSPRIVATE_SETSTRINGPREF = 987,
  DELETED_SETTINGSPRIVATE_SETURLPREF = 988,
  SETTINGSPRIVATE_GETALLPREFS = 989,
  SETTINGSPRIVATE_GETPREF = 990,
  NETWORKINGPRIVATE_FORGETNETWORK = 991,
  DELETED_EASYUNLOCKPRIVATE_HIDEERRORBUBBLE = 992,
  WEBVIEWINTERNAL_SETZOOMMODE = 993,
  WEBVIEWINTERNAL_GETZOOMMODE = 994,
  LAUNCHERSEARCHPROVIDER_SETSEARCHRESULTS = 995,
  DATAREDUCTIONPROXY_CLEARDATASAVINGS = 996,
  BLUETOOTHPRIVATE_SETDISCOVERYFILTER = 997,
  FILESYSTEM_GETVOLUMELIST = 998,
  NETWORKINGPRIVATE_GETDEVICESTATES = 999,
  DEVELOPERPRIVATE_GETPROFILECONFIGURATION = 1000,
  DEVELOPERPRIVATE_UPDATEPROFILECONFIGURATION = 1001,
  SOCKETS_UDP_SETBROADCAST = 1002,
  FILEMANAGERPRIVATE_GETPROVIDERS = 1003,
  WEBVIEWINTERNAL_ADDCONTENTSCRIPTS = 1004,
  WEBVIEWINTERNAL_REMOVECONTENTSCRIPTS = 1005,
  DEVELOPERPRIVATE_REPAIREXTENSION = 1006,
  DEVELOPERPRIVATE_SHOWOPTIONS = 1007,
  DEVELOPERPRIVATE_SHOWPATH = 1008,
  FILEMANAGERPRIVATE_ADDPROVIDEDFILESYSTEM = 1009,
  FILEMANAGERPRIVATE_CONFIGUREVOLUME = 1010,
  DELETED_SEARCHENGINESPRIVATE_GETSEARCHENGINES = 1011,
  DELETED_SEARCHENGINESPRIVATE_SETSELECTEDSEARCHENGINE = 1012,
  AUTOFILLPRIVATE_SAVEADDRESS = 1013,
  AUTOFILLPRIVATE_GETADDRESSCOMPONENTS = 1014,
  AUTOFILLPRIVATE_SAVECREDITCARD = 1015,
  AUTOFILLPRIVATE_REMOVEENTRY = 1016,
  AUTOFILLPRIVATE_VALIDATEPHONENUMBERS = 1017,
  AUTOFILLPRIVATE_MASKCREDITCARD = 1018,
  INSTANCEID_GETID = 1019,
  INSTANCEID_GETCREATIONTIME = 1020,
  INSTANCEID_GETTOKEN = 1021,
  INSTANCEID_DELETETOKEN = 1022,
  INSTANCEID_DELETEID = 1023,
  DELETED_GCDPRIVATE_GETDEVICEINFO = 1024,
  DELETED_SEARCHENGINESPRIVATE_GETHOTWORDSTATE = 1025,
  DELETED_SEARCHENGINESPRIVATE_ADDOTHERSEARCHENGINE = 1026,
  DELETED_SEARCHENGINESPRIVATE_REMOVESEARCHENGINE = 1027,
  DELETED_SEARCHENGINESPRIVATE_UPDATESEARCHENGINE = 1028,
  DELETED_SEARCHENGINESPRIVATE_OPTINTOHOTWORDING = 1029,
  WEBSTOREWIDGETPRIVATE_INSTALLWEBSTOREITEM = 1030,
  DELETED_WEBSTOREWIDGETPRIVATE_GETSTRINGS = 1031,
  WEBRTCDESKTOPCAPTUREPRIVATE_CHOOSEDESKTOPMEDIA = 1032,
  WEBRTCDESKTOPCAPTUREPRIVATE_CANCELCHOOSEDESKTOPMEDIA = 1033,
  BLUETOOTHLOWENERGY_REGISTERADVERTISEMENT = 1034,
  BLUETOOTHLOWENERGY_UNREGISTERADVERTISEMENT = 1035,
  DELETED_PASSWORDSPRIVATE_CANPASSWORDACCOUNTBEMANAGED = 1036,
  PASSWORDSPRIVATE_REMOVESAVEDPASSWORD = 1037,
  PASSWORDSPRIVATE_REMOVEPASSWORDEXCEPTION = 1038,
  PASSWORDSPRIVATE_REQUESTPLAINTEXTPASSWORD = 1039,
  DELETED_LAUNCHERPAGE_HIDE = 1040,
  PLATFORMKEYS_VERIFYTLSSERVERCERTIFICATE = 1041,
  DEVELOPERPRIVATE_SETSHORTCUTHANDLINGSUSPENDED = 1042,
  DEVELOPERPRIVATE_UPDATEEXTENSIONCOMMAND = 1043,
  INPUTMETHODPRIVATE_GETENCRYPTSYNCENABLED = 1044,
  USERSPRIVATE_GETWHITELISTEDUSERS = 1045,
  USERSPRIVATE_ADDWHITELISTEDUSER = 1046,
  USERSPRIVATE_REMOVEWHITELISTEDUSER = 1047,
  USERSPRIVATE_ISCURRENTUSEROWNER = 1048,
  USERSPRIVATE_ISWHITELISTMANAGED = 1049,
  PRINTERPROVIDERINTERNAL_REPORTUSBPRINTERINFO = 1050,
  WEBCAMPRIVATE_OPENSERIALWEBCAM = 1051,
  WEBCAMPRIVATE_CLOSEWEBCAM = 1052,
  SERIAL_SETBREAK = 1053,
  SERIAL_CLEARBREAK = 1054,
  DELETED_EXTENSIONVIEWINTERNAL_LOADSRC = 1055,   // Obsolete: crbug.com/982858
  DELETED_EXTENSIONVIEWINTERNAL_PARSESRC = 1056,  // Obsolete: crbug.com/982858
  HID_GETUSERSELECTEDDEVICES = 1057,
  FILESYSTEMPROVIDERINTERNAL_GETACTIONSREQUESTEDSUCCESS = 1058,
  DASHBOARDPRIVATE_SHOWPERMISSIONPROMPTFORDELEGATEDBUNDLEINSTALL = 1059,
  FILEMANAGERPRIVATEINTERNAL_GETCUSTOMACTIONS = 1060,
  FILEMANAGERPRIVATEINTERNAL_EXECUTECUSTOMACTION = 1061,
  MDNS_FORCEDISCOVERY = 1062,
  LANGUAGESETTINGSPRIVATE_GETLANGUAGELIST = 1063,
  DELETED_LANGUAGESETTINGSPRIVATE_SETLANGUAGELIST = 1064,
  LANGUAGESETTINGSPRIVATE_GETSPELLCHECKDICTIONARYSTATUS = 1065,
  LANGUAGESETTINGSPRIVATE_GETSPELLCHECKWORDS = 1066,
  LANGUAGESETTINGSPRIVATE_GETTRANSLATETARGETLANGUAGE = 1067,
  LANGUAGESETTINGSPRIVATE_GETINPUTMETHODLISTS = 1068,
  LANGUAGESETTINGSPRIVATE_ADDINPUTMETHOD = 1069,
  LANGUAGESETTINGSPRIVATE_REMOVEINPUTMETHOD = 1070,
  FILEMANAGERPRIVATE_CANCELALLFILETRANSFERS = 1071,
  NETWORKINGPRIVATE_UNLOCKCELLULARSIM = 1072,
  NETWORKINGPRIVATE_SETCELLULARSIMSTATE = 1073,
  ENTERPRISE_DEVICEATTRIBUTES_GETDIRECTORYDEVICEID = 1074,
  CASTDEVICESPRIVATE_UPDATEDEVICES = 1075,
  SYSTEM_DISPLAY_ENABLEUNIFIEDDESKTOP = 1076,
  BROWSINGDATA_REMOVECACHESTORAGE = 1077,
  VIRTUALKEYBOARDPRIVATE_SETKEYBOARDSTATE = 1078,
  VIRTUALKEYBOARDPRIVATE_SETHOTRODKEYBOARD = 1079,
  CERTIFICATEPROVIDERINTERNAL_REPORTSIGNATURE = 1080,
  CERTIFICATEPROVIDERINTERNAL_REPORTCERTIFICATES = 1081,
  BLUETOOTHPRIVATE_PAIR = 1082,
  DELETED_EASYUNLOCKPRIVATE_FINDSETUPCONNECTION = 1083,
  RESOURCESPRIVATE_GETSTRINGS = 1084,
  USB_GETCONFIGURATIONS = 1085,
  DELETED_EASYUNLOCKPRIVATE_SETUPCONNECTIONSTATUS = 1086,
  DELETED_EASYUNLOCKPRIVATE_SETUPCONNECTIONDISCONNECT = 1087,
  DELETED_EASYUNLOCKPRIVATE_SETUPCONNECTIONSEND = 1088,
  DATAREDUCTIONPROXY_GETDATAUSAGE = 1089,
  DELETED_EASYUNLOCKPRIVATE_SETUPCONNECTIONGETDEVICEADDRESS = 1090,
  TABCAPTURE_CAPTUREOFFSCREENTAB = 1091,
  LANGUAGESETTINGSPRIVATE_ADDSPELLCHECKWORD = 1092,
  LANGUAGESETTINGSPRIVATE_REMOVESPELLCHECKWORD = 1093,
  SETTINGSPRIVATE_GETDEFAULTZOOMFUNCTION = 1094,
  SETTINGSPRIVATE_SETDEFAULTZOOMFUNCTION = 1095,
  BLUETOOTHPRIVATE_CONNECT = 1096,
  BLUETOOTHPRIVATE_FORGETDEVICE = 1097,
  DISPLAYSOURCE_GETAVAILABLESINKS = 1098,
  DISPLAYSOURCE_REQUESTAUTHENTICATION = 1099,
  WEBRTCLOGGINGPRIVATE_STARTAUDIODEBUGRECORDINGS = 1100,
  WEBRTCLOGGINGPRIVATE_STOPAUDIODEBUGRECORDINGS = 1101,
  TERMINALPRIVATE_ACKOUTPUT = 1102,
  INPUT_IME_CREATEWINDOW = 1103,
  INPUTMETHODPRIVATE_SETXKBLAYOUT = 1104,
  ACCESSIBILITY_PRIVATE_SETKEYBOARDLISTENER = 1105,
  INPUT_IME_ACTIVATE = 1106,
  INPUT_IME_DEACTIVATE = 1107,
  ENTERPRISE_PLATFORMKEYS_CHALLENGEMACHINEKEY = 1108,
  ENTERPRISE_PLATFORMKEYS_CHALLENGEUSERKEY = 1109,
  INPUTMETHODPRIVATE_NOTIFYIMEMENUITEMACTIVATED = 1110,
  INPUT_IME_SHOWWINDOW = 1111,
  INPUT_IME_HIDEWINDOW = 1112,
  INPUTMETHODPRIVATE_SHOWINPUTVIEW = 1113,
  WALLPAPERPRIVATE_RECORDWALLPAPERUMA = 1114,
  AUTOTESTPRIVATE_GETVISIBLENOTIFICATIONS = 1115,
  DELETED_WEBRTCLOGGINGPRIVATE_STARTRTCEVENTLOGGING = 1116,
  DELETED_WEBRTCLOGGINGPRIVATE_STOPRTCEVENTLOGGING = 1117,
  PASSWORDSPRIVATE_GETSAVEDPASSWORDLIST = 1118,
  PASSWORDSPRIVATE_GETPASSWORDEXCEPTIONLIST = 1119,
  INPUTMETHODPRIVATE_OPENOPTIONSPAGE = 1120,
  DELETED_FEEDBACKPRIVATE_LOGSRTPROMPTRESULT = 1121,
  BLUETOOTHLOWENERGY_CREATESERVICE = 1122,
  BLUETOOTHLOWENERGY_CREATECHARACTERISTIC = 1123,
  BLUETOOTHLOWENERGY_CREATEDESCRIPTOR = 1124,
  BLUETOOTHLOWENERGY_REGISTERSERVICE = 1125,
  BLUETOOTHLOWENERGY_UNREGISTERSERVICE = 1126,
  BLUETOOTHLOWENERGY_SENDREQUESTRESPONSE = 1127,
  BLUETOOTHLOWENERGY_NOTIFYCHARACTERISTICVALUECHANGED = 1128,
  BLUETOOTHLOWENERGY_REMOVESERVICE = 1129,
  AUTOFILLPRIVATE_GETADDRESSLIST = 1130,
  AUTOFILLPRIVATE_GETCREDITCARDLIST = 1131,
  SYSTEM_DISPLAY_OVERSCANCALIBRATIONSTART = 1132,
  SYSTEM_DISPLAY_OVERSCANCALIBRATIONADJUST = 1133,
  SYSTEM_DISPLAY_OVERSCANCALIBRATIONRESET = 1134,
  SYSTEM_DISPLAY_OVERSCANCALIBRATIONCOMPLETE = 1135,
  AUTOFILLPRIVATE_GETCOUNTRYLIST = 1136,
  RUNTIME_RESTARTAFTERDELAY = 1137,
  SYSTEM_DISPLAY_SETDISPLAYLAYOUT = 1138,
  SYSTEM_DISPLAY_GETDISPLAYLAYOUT = 1139,
  QUICKUNLOCKPRIVATE_SETMODES = 1140,
  QUICKUNLOCKPRIVATE_GETACTIVEMODES = 1141,
  QUICKUNLOCKPRIVATE_GETAVAILABLEMODES = 1142,
  WEBSTOREPRIVATE_ISPENDINGCUSTODIANAPPROVAL = 1143,
  TABS_DISCARD = 1144,
  LANGUAGESETTINGSPRIVATE_ENABLELANGUAGE = 1145,
  LANGUAGESETTINGSPRIVATE_DISABLELANGUAGE = 1146,
  FILEMANAGERPRIVATEINTERNAL_GETDIRECTORYSIZE = 1147,
  BLUETOOTHLOWENERGY_SETADVERTISINGINTERVAL = 1148,
  SYSTEM_DISPLAY_SHOWNATIVETOUCHCALIBRATION = 1149,
  SYSTEM_DISPLAY_COMPLETECUSTOMTOUCHCALIBRATION = 1150,
  SYSTEM_DISPLAY_CLEARTOUCHCALIBRATION = 1151,
  CERTIFICATEPROVIDER_REQUESTPIN = 1152,
  CERTIFICATEPROVIDER_STOPPINREQUEST = 1153,
  QUICKUNLOCKPRIVATE_CHECKCREDENTIAL = 1154,
  QUICKUNLOCKPRIVATE_GETCREDENTIALREQUIREMENTS = 1155,
  CLIPBOARD_SETIMAGEDATA = 1156,
  NETWORKINGPRIVATE_GETGLOBALPOLICY = 1157,
  DELETED_DIAL_FETCHDEVICEDESCRIPTION = 1158,
  SYSTEM_DISPLAY_STARTCUSTOMTOUCHCALIBRATION = 1159,
  AUDIO_GETMUTE = 1160,
  AUDIO_SETMUTE = 1161,
  AUDIO_GETDEVICES = 1162,
  VIRTUALKEYBOARD_RESTRICTFEATURES = 1163,
  NETWORKINGCASTPRIVATE_VERIFYDESTINATION = 1164,
  NETWORKINGCASTPRIVATE_VERIFYANDENCRYPTCREDENTIALS = 1165,
  NETWORKINGCASTPRIVATE_VERIFYANDENCRYPTDATA = 1166,
  NETWORKINGCASTPRIVATE_SETWIFITDLSENABLEDSTATE = 1167,
  NETWORKINGCASTPRIVATE_GETWIFITDLSSTATUS = 1168,
  ACCESSIBILITY_PRIVATE_DARKENSCREEN = 1169,
  // This does not follow the naming convention, but follows the other APIs in
  // webrtcAudioPrivate.
  WEBRTC_AUDIO_PRIVATE_SET_AUDIO_EXPERIMENTS = 1170,
  AUTOTESTPRIVATE_GETPLAYSTORESTATE = 1171,
  AUTOTESTPRIVATE_SETPLAYSTOREENABLED = 1172,
  APP_CURRENTWINDOWINTERNAL_SETACTIVATEONPOINTER = 1173,
  MEDIAPERCEPTIONPRIVATE_GETSTATE = 1174,
  MEDIAPERCEPTIONPRIVATE_SETSTATE = 1175,
  MEDIAPERCEPTIONPRIVATE_GETDIAGNOSTICS = 1176,
  NETWORKINGPRIVATE_GETCERTIFICATELISTS = 1177,
  DELETED_ACCESSIBILITY_PRIVATE_SETSWITCHACCESSKEYS = 1178,
  FEEDBACKPRIVATE_READLOGSOURCE = 1179,
  LOCKSCREENDATA_CREATE = 1180,
  LOCKSCREENDATA_GETALL = 1181,
  LOCKSCREENDATA_GETCONTENT = 1182,
  LOCKSCREENDATA_SETCONTENT = 1183,
  LOCKSCREENDATA_DELETE = 1184,
  BLUETOOTHLOWENERGY_RESETADVERTISING = 1185,
  WEBVIEWINTERNAL_SETAUDIOMUTED = 1186,
  WEBVIEWINTERNAL_ISAUDIOMUTED = 1187,
  WEBVIEWINTERNAL_GETAUDIOSTATE = 1188,
  FILEMANAGERPRIVATE_GETRECENTFILES = 1189,
  FILEMANAGERPRIVATE_RENAMEVOLUME = 1190,
  AUTOTESTPRIVATE_SETMOUSEREVERSESCROLL = 1191,
  METRICSPRIVATE_RECORDSPARSEHASHABLE = 1192,
  NETWORKINGPRIVATE_SELECTCELLULARMOBILENETWORK = 1193,
  PASSWORDSPRIVATE_IMPORTPASSWORDS = 1194,
  PASSWORDSPRIVATE_EXPORTPASSWORDS = 1195,
  PASSWORDSPRIVATE_UNDOREMOVESAVEDPASSWORDOREXCEPTION = 1196,
  ACCESSIBILITY_PRIVATE_SETNATIVECHROMEVOXARCSUPPORTFORCURRENTAPP = 1197,
  LANGUAGESETTINGSPRIVATE_SETENABLETRANSLATIONFORLANGUAGE = 1198,
  LANGUAGESETTINGSPRIVATE_MOVELANGUAGE = 1199,
  MEDIAPERCEPTIONPRIVATE_SETANALYTICSCOMPONENT = 1200,
  ACCESSIBILITY_PRIVATE_SETHIGHLIGHTS = 1201,
  WEBRTCLOGGINGPRIVATE_GETLOGSDIRECTORY = 1202,
  VIRTUALKEYBOARDPRIVATE_SETDRAGGABLEAREA = 1203,
  METRICSPRIVATE_RECORDBOOLEAN = 1204,
  METRICSPRIVATE_RECORDENUMERATIONVALUE = 1205,
  DEVELOPERPRIVATE_NOTIFYDRAGINSTALLINPROGRESS = 1206,
  AUTOTESTPRIVATE_GETPRINTERLIST = 1207,
  DEVELOPERPRIVATE_GETEXTENSIONSIZE = 1208,
  CRYPTOTOKENPRIVATE_ISAPPIDHASHINENTERPRISECONTEXT = 1209,
  CRYPTOTOKENPRIVATE_CANAPPIDGETATTESTATION = 1210,
  SYSTEM_DISPLAY_SETMIRRORMODE = 1211,
  TTSENGINE_UPDATEVOICES = 1212,
  PASSWORDSPRIVATE_REQUESTEXPORTPROGRESSSTATUS = 1213,
  WALLPAPERPRIVATE_GETCOLLECTIONSINFO = 1214,
  WALLPAPERPRIVATE_GETIMAGESINFO = 1215,
  ACCESSIBILITY_PRIVATE_SENDSYNTHETICKEYEVENT = 1216,
  WALLPAPERPRIVATE_GETLOCALIMAGEPATHS = 1217,
  WALLPAPERPRIVATE_GETLOCALIMAGEDATA = 1218,
  ENTERPRISE_DEVICEATTRIBUTES_GETDEVICESERIALNUMBER = 1219,
  ENTERPRISE_DEVICEATTRIBUTES_GETDEVICEASSETID = 1220,
  ENTERPRISE_DEVICEATTRIBUTES_GETDEVICEANNOTATEDLOCATION = 1221,
  PASSWORDSPRIVATE_CANCELEXPORTPASSWORDS = 1222,
  FILEMANAGERPRIVATE_MARKCACHEASMOUNTED = 1223,
  WALLPAPERPRIVATE_CONFIRMPREVIEWWALLPAPER = 1224,
  WALLPAPERPRIVATE_CANCELPREVIEWWALLPAPER = 1225,
  WEBRTCLOGGINGPRIVATE_STARTEVENTLOGGING = 1226,
  VIRTUALKEYBOARDPRIVATE_SETCONTAINERBEHAVIOR = 1227,
  QUICKUNLOCKPRIVATE_GETAUTHTOKEN = 1228,
  QUICKUNLOCKPRIVATE_SETLOCKSCREENENABLED = 1229,
  LANGUAGESETTINGSPRIVATE_RETRYDOWNLOADDICTIONARY = 1230,
  DELETED_DECLARATIVENETREQUEST_ADDALLOWEDPAGES = 1231,
  DELETED_DECLARATIVENETREQUEST_REMOVEALLOWEDPAGES = 1232,
  DELETED_DECLARATIVENETREQUEST_GETALLOWEDPAGES = 1233,
  DEVELOPERPRIVATE_INSTALLDROPPEDFILE = 1234,
  AUTOMATIONINTERNAL_ENABLETREE = 1235,
  AUTOMATIONINTERNAL_QUERYSELECTOR = 1236,
  DEBUGGER_GETTARGETS = 1237,
  NOTIFICATIONS_GETPERMISSIONLEVEL = 1238,
  DELETED_FILEMANAGERPRIVATE_ENSUREFILEDOWNLOADED = 1239,
  FILEMANAGERPRIVATE_OPENSETTINGSSUBPAGE = 1240,
  ENTERPRISEREPORTINGPRIVATE_UPLOADCHROMEDESKTOPREPORT = 1241,
  CECPRIVATE_SENDSTANDBY = 1242,
  CECPRIVATE_SENDWAKEUP = 1243,
  WEBSTOREPRIVATE_GETREFERRERCHAIN = 1244,
  AUTOTESTPRIVATE_UPDATEPRINTER = 1245,
  AUTOTESTPRIVATE_REMOVEPRINTER = 1246,
  WALLPAPERPRIVATE_GETCURRENTWALLPAPERTHUMBNAIL = 1247,
  ACCESSIBILITY_PRIVATE_ONSELECTTOSPEAKSTATECHANGED = 1248,
  INPUTMETHODPRIVATE_GETCOMPOSITIONBOUNDS = 1249,
  DELETED_FILEMANAGERPRIVATE_ISCROSTINIENABLED = 1250,
  FILEMANAGERPRIVATE_MOUNTCROSTINI = 1251,
  CECPRIVATE_QUERYDISPLAYCECPOWERSTATE = 1252,
  DEVELOPERPRIVATE_ADDHOSTPERMISSION = 1253,
  DEVELOPERPRIVATE_REMOVEHOSTPERMISSION = 1254,
  MEDIAPERCEPTIONPRIVATE_SETCOMPONENTPROCESSSTATE = 1255,
  USERSPRIVATE_GETCURRENTUSER = 1256,
  WALLPAPERPRIVATE_GETSURPRISEMEIMAGE = 1257,
  VIRTUALKEYBOARDPRIVATE_SETOCCLUDEDBOUNDS = 1258,
  SYSTEM_POWER_SOURCE_GETPOWERSOURCEINFO = 1259,
  SYSTEM_POWER_SOURCE_REQUESTSTATUSUPDATE = 1260,
  INPUTMETHODPRIVATE_GETSURROUNDINGTEXT = 1261,
  USERSPRIVATE_GETLOGINSTATUS = 1262,
  FILEMANAGERPRIVATEINTERNAL_INSTALLLINUXPACKAGE = 1263,
  VIRTUALKEYBOARDPRIVATE_SETHITTESTBOUNDS = 1264,
  ARCAPPSPRIVATE_GETLAUNCHABLEAPPS = 1265,
  ARCAPPSPRIVATE_LAUNCHAPP = 1266,
  AUTOTESTPRIVATE_RUNCROSTINIINSTALLER = 1267,
  AUTOFILLPRIVATE_MIGRATECREDITCARDS = 1268,
  AUTOTESTPRIVATE_ISAPPSHOWN = 1269,
  ENTERPRISEREPORTINGPRIVATE_GETDEVICEID = 1270,
  ENTERPRISE_HARDWAREPLATFORM_GETHARDWAREPLATFORMINFO = 1271,
  FILEMANAGERPRIVATEINTERNAL_SHAREPATHSWITHCROSTINI = 1272,
  AUTOTESTPRIVATE_SETCROSTINIENABLED = 1273,
  METRICSPRIVATE_GETHISTOGRAM = 1274,
  TABCAPTURE_GETMEDIASTREAMID = 1275,
  WEBVIEWINTERNAL_SETSPATIALNAVIGATIONENABLED = 1276,
  WEBVIEWINTERNAL_ISSPATIALNAVIGATIONENABLED = 1277,
  FILEMANAGERPRIVATEINTERNAL_GETTHUMBNAIL = 1278,
  FILEMANAGERPRIVATEINTERNAL_GETCROSTINISHAREDPATHS = 1279,
  AUTOTESTPRIVATE_LAUNCHAPP = 1280,
  AUTOTESTPRIVATE_BOOTSTRAPMACHINELEARNINGSERVICE = 1281,
  AUTOTESTPRIVATE_RUNCROSTINIUNINSTALLER = 1282,
  AUTOTESTPRIVATE_TAKESCREENSHOT = 1283,
  ACCESSIBILITY_PRIVATE_TOGGLEDICTATION = 1284,
  ACCESSIBILITY_PRIVATE_ENABLECHROMEVOXMOUSEEVENTS = 1285,
  ACCESSIBILITY_PRIVATE_SENDSYNTHETICMOUSEEVENT = 1286,
  FILEMANAGERPRIVATE_DETECTCHARACTERENCODING = 1287,
  FILEMANAGERPRIVATEINTERNAL_GETLINUXPACKAGEINFO = 1288,
  TABS_GOFORWARD = 1289,
  TABS_GOBACK = 1290,
  BRAILLEDISPLAYPRIVATE_UPDATEBLUETOOTHBRAILLEDISPLAYADDRESS = 1291,
  AUTOTESTPRIVATE_SETASSISTANTENABLED = 1292,
  AUTOTESTPRIVATE_ISARCPROVISIONED = 1293,
  CRYPTOTOKENPRIVATE_CANPROXYTOWEBAUTHN = 1294,
  INPUTMETHODPRIVATE_GETSETTINGS = 1295,
  INPUTMETHODPRIVATE_SETSETTINGS = 1296,
  FILEMANAGERPRIVATEINTERNAL_UNSHAREPATHWITHCROSTINI = 1297,
  PASSWORDSPRIVATE_RECORDPASSWORDSPAGEACCESSINSETTINGS = 1298,
  AUTOFILLPRIVATE_SERVERCARDLINKCLICKED = 1299,
  AUTOTESTPRIVATE_GETARCAPP = 1300,
  AUTOTESTPRIVATE_GETARCPACKAGE = 1301,
  AUTOTESTPRIVATE_LAUNCHARCAPP = 1302,
  AUTOTESTPRIVATE_CLOSEAPP = 1303,
  ACCESSIBILITY_PRIVATE_SETSWITCHACCESSMENUSTATE = 1304,
  AUTOTESTPRIVATE_SENDASSISTANTTEXTQUERY = 1305,
  AUTOTESTPRIVATE_SETCROSTINIAPPSCALED = 1306,
  ACTIVITYLOGPRIVATE_DELETEACTIVITIESBYEXTENSION = 1307,
  ACCESSIBILITY_PRIVATE_FORWARDKEYEVENTSTOSWITCHACCESS = 1308,
  AUTOTESTPRIVATE_ENSUREWINDOWSERVICECLIENTHASDRAWNWINDOW = 1309,
  ACCESSIBILITY_PRIVATE_GETBATTERYDESCRIPTION = 1310,
  IDLE_GETAUTOLOCKDELAY = 1311,
  AUTOTESTPRIVATE_GETPRIMARYDISPLAYSCALEFACTOR = 1312,
  AUTOFILLPRIVATE_GETLOCALCREDITCARDLIST_DEPRECATED = 1313,
  AUTOFILLPRIVATE_GETSERVERCREDITCARDLIST_DEPRECATED = 1314,
  PASSWORDSPRIVATE_CHANGESAVEDPASSWORD = 1315,
  AUTOTESTPRIVATE_SETWHITELISTEDPREF = 1316,
  SAFEBROWSINGPRIVATE_GETREFERRERCHAIN = 1317,
  DELETED_DECLARATIVENETREQUEST_ADDDYNAMICRULES = 1318,
  DELETED_DECLARATIVENETREQUEST_REMOVEDYNAMICRULES = 1319,
  DECLARATIVENETREQUEST_GETDYNAMICRULES = 1320,
  AUTOTESTPRIVATE_GETARCSTATE = 1321,
  AUTOTESTPRIVATE_ISTABLETMODEENABLED = 1322,
  AUTOTESTPRIVATE_SETTABLETMODEENABLED = 1323,
  AUTOTESTPRIVATE_GETSHELFAUTOHIDEBEHAVIOR = 1324,
  AUTOTESTPRIVATE_SETSHELFAUTOHIDEBEHAVIOR = 1325,
  AUTOTESTPRIVATE_EXPORTCROSTINI = 1326,
  AUTOTESTPRIVATE_IMPORTCROSTINI = 1327,
  ACCESSIBILITY_PRIVATE_SETVIRTUALKEYBOARDVISIBLE = 1328,
  AUTOTESTPRIVATE_SHOWVIRTUALKEYBOARDIFENABLED = 1329,
  FEEDBACKPRIVATE_LOGINFEEDBACKCOMPLETE = 1330,
  FILEMANAGERPRIVATE_SEARCHFILES = 1331,
  MANAGEMENT_INSTALLREPLACEMENTWEBAPP = 1332,
  FILEMANAGERPRIVATE_GETANDROIDPICKERAPPS = 1333,
  FILEMANAGERPRIVATE_SELECTANDROIDPICKERAPP = 1334,
  AUTOTESTPRIVATE_GETSHELFALIGNMENT = 1335,
  AUTOTESTPRIVATE_SETSHELFALIGNMENT = 1336,
  BLUETOOTHPRIVATE_RECORDPAIRING = 1337,
  FILEMANAGERPRIVATE_SETARCSTORAGETOASTSHOWNFLAG = 1338,
  INPUTMETHODPRIVATE_SETCOMPOSITIONRANGE = 1339,
  BLUETOOTHPRIVATE_RECORDRECONNECTION = 1340,
  LOGINSCREENUI_SHOW = 1341,
  LOGINSCREENUI_CLOSE = 1342,
  DECLARATIVENETREQUEST_GETMATCHEDRULES = 1343,
  DECLARATIVENETREQUEST_SETACTIONCOUNTASBADGETEXT = 1344,
  BLUETOOTHPRIVATE_RECORDDEVICESELECTION = 1345,
  LOGIN_LAUNCHMANAGEDGUESTSESSION = 1346,
  LOGIN_EXITCURRENTSESSION = 1347,
  DELETED_LOGIN_ISRUNNINGINLOGINPROFILE = 1348,
  LOGIN_FETCHDATAFORNEXTLOGINATTEMPT = 1349,
  ACTION_SETICON = 1350,
  AUTOTESTPRIVATE_GETALLENTERPRISEPOLICIES = 1351,
  ACCESSIBILITY_PRIVATE_ONSCROLLABLEBOUNDSFORPOINTFOUND = 1352,
  ACTION_GETPOPUP = 1353,
  ACTION_SETPOPUP = 1354,
  ACTION_GETTITLE = 1355,
  ACTION_SETTITLE = 1356,
  ACTION_GETBADGETEXT = 1357,
  ACTION_SETBADGETEXT = 1358,
  ACTION_GETBADGEBACKGROUNDCOLOR = 1359,
  ACTION_SETBADGEBACKGROUNDCOLOR = 1360,
  DELETED_AUTOTESTPRIVATE_SETARCAPPWINDOWSTATE = 1361,
  ACCESSIBILITY_PRIVATE_OPENSETTINGSSUBPAGE = 1362,
  ACTION_ENABLE = 1363,
  ACTION_DISABLE = 1364,
  FILEMANAGERPRIVATEINTERNAL_IMPORTCROSTINIIMAGE = 1365,
  AUTOTESTPRIVATE_GETSHELFITEMS = 1366,
  MANAGEMENT_INSTALLREPLACEMENTANDROIDAPP = 1367,
  MANAGEMENT_CANINSTALLREPLACEMENTANDROIDAPP = 1368,
  AUTOTESTPRIVATE_REGISTERCOMPONENT = 1369,
  LOGINSCREENSTORAGE_STOREPERSISTENTDATA = 1370,
  LOGINSCREENSTORAGE_RETRIEVEPERSISTENTDATA = 1371,
  LOGINSCREENSTORAGE_STORECREDENTIALS = 1372,
  LOGINSCREENSTORAGE_RETRIEVECREDENTIALS = 1373,
  DELETED_AUTOTESTPRIVATE_GETARCAPPWINDOWSTATE = 1374,
  DELETED_AUTOTESTPRIVATE_GETARCAPPWINDOWINFO = 1375,
  AUTOTESTPRIVATE_GETALLINSTALLEDAPPS = 1376,
  AUTOTESTPRIVATE_SWAPWINDOWSINSPLITVIEW = 1377,
  AUTOTESTPRIVATE_SETARCAPPWINDOWFOCUS = 1378,
  LOGINSTATE_GETPROFILETYPE = 1379,
  LOGINSTATE_GETSESSIONSTATE = 1380,
  AUTOTESTPRIVATE_GETARCSTARTTIME = 1381,
  AUTOTESTPRIVATE_SETOVERVIEWMODESTATE = 1382,
  AUTOTESTPRIVATE_TAKESCREENSHOTFORDISPLAY = 1383,
  AUTOFILLPRIVATE_SETCREDITCARDFIDOAUTHENABLEDSTATE = 1384,
  USERSPRIVATE_ISWHITELISTEDUSER = 1385,
  PRINTINGMETRICS_GETPRINTJOBS = 1386,
  AUTOTESTPRIVATE_WAITFORASSISTANTQUERYSTATUS = 1387,
  AUTOTESTPRIVATE_GETCLIPBOARDTEXTDATA = 1388,
  AUTOTESTPRIVATE_SETCLIPBOARDTEXTDATA = 1389,
  AUTOTESTPRIVATE_INITIALIZEEVENTS = 1390,
  AUTOTESTPRIVATE_WAITFORDISPLAYROTATION = 1391,
  AUTOTESTPRIVATE_ARCAPPTRACINGSTART = 1392,
  AUTOTESTPRIVATE_ARCAPPTRACINGSTOPANDANALYZE = 1393,
  AUTOTESTPRIVATE_GETAPPWINDOWLIST = 1394,
  AUTOTESTPRIVATE_SETAPPWINDOWSTATE = 1395,
  AUTOTESTPRIVATE_CLOSEAPPWINDOW = 1396,
  AUTOTESTPRIVATE_REFRESHENTERPRISEPOLICIES = 1397,
  AUTOTESTPRIVATE_INSTALLPWAFORCURRENTURL = 1398,
  AUTOTESTPRIVATE_ACTIVATEACCELERATOR = 1399,
  AUTOTESTPRIVATE_WAITFORLAUNCHERSTATE = 1400,
  AUTOTESTPRIVATE_CREATENEWDESK = 1401,
  AUTOTESTPRIVATE_ACTIVATEDESKATINDEX = 1402,
  AUTOTESTPRIVATE_REMOVEACTIVEDESK = 1403,
  TERMINALPRIVATE_GETCROSHSETTINGS = 1404,
  AUTOTESTPRIVATE_ENABLEASSISTANTANDWAITFORREADY = 1405,
  INPUTMETHODPRIVATE_FINISHCOMPOSINGTEXT = 1406,
  DECLARATIVENETREQUEST_UPDATEDYNAMICRULES = 1407,
  AUTOTESTPRIVATE_MOUSECLICK = 1408,
  AUTOTESTPRIVATE_MOUSEPRESS = 1409,
  AUTOTESTPRIVATE_MOUSERELEASE = 1410,
  AUTOTESTPRIVATE_MOUSEMOVE = 1411,
  WEBSTOREPRIVATE_GETEXTENSIONSTATUS = 1412,
  INPUTMETHODPRIVATE_SETSELECTIONRANGE = 1413,
  VIRTUALKEYBOARDPRIVATE_SETAREATOREMAINONSCREEN = 1414,
  TERMINALPRIVATE_GETSETTINGS = 1415,
  TERMINALPRIVATE_SETSETTINGS = 1416,
  WEBSTOREPRIVATE_REQUESTEXTENSION = 1417,
  AUTOTESTPRIVATE_INSTALLPLUGINVM = 1418,
  CRASHREPORTPRIVATE_REPORTERROR = 1419,
  AUTOTESTPRIVATE_SETMETRICSENABLED = 1420,
  ENTERPRISEREPORTINGPRIVATE_GETPERSISTENTSECRET = 1421,
  ENTERPRISEREPORTINGPRIVATE_GETDEVICEDATA = 1422,
  ENTERPRISEREPORTINGPRIVATE_SETDEVICEDATA = 1423,
  ENTERPRISEREPORTINGPRIVATE_GETDEVICEINFO = 1424,
  PRINTING_GETPRINTERS = 1425,
  WEBCAMPRIVATE_SET_HOME = 1426,
  INPUTMETHODPRIVATE_RESET = 1427,
  PRINTING_GETPRINTERINFO = 1428,
  AUTOTESTPRIVATE_ISARCPACKAGELISTINITIALREFRESHED = 1429,
  AUTOTESTPRIVATE_STARTTRACING = 1430,
  AUTOTESTPRIVATE_STOPTRACING = 1431,
  LOGIN_LOCKMANAGEDGUESTSESSION = 1432,
  LOGIN_UNLOCKMANAGEDGUESTSESSION = 1433,
  AUTOTESTPRIVATE_SETARCTOUCHMODE = 1434,
  PRINTING_SUBMITJOB = 1435,
  IDENTITYPRIVATE_SETCONSENTRESULT = 1436,
  PRINTING_CANCELJOB = 1437,
  AUTOFILLASSISTANTPRIVATE_CREATE = 1438,
  AUTOFILLASSISTANTPRIVATE_START = 1439,
  AUTOFILLASSISTANTPRIVATE_GETSTATE = 1440,
  AUTOFILLASSISTANTPRIVATE_PERFORMACTION = 1441,
  AUTOFILLASSISTANTPRIVATE_PROVIDEUSERDATA = 1442,
  PASSWORDSPRIVATE_ISOPTEDINFORACCOUNTSTORAGE = 1443,
  AUTOTESTPRIVATE_PINSHELFICON = 1444,
  AUTOTESTPRIVATE_WAITFOROVERVIEWSTATE = 1445,
  AUTOTESTPRIVATE_GETSCROLLABLESHELFINFOFORSTATE = 1446,
  ENTERPRISE_DEVICEATTRIBUTES_GETDEVICEHOSTNAME = 1447,
  AUTOFILLPRIVATE_GETUPIIDLIST = 1448,
  WEBCAMPRIVATE_RESTORE_CAMERA_PRESET = 1449,
  WEBCAMPRIVATE_SET_CAMERA_PRESET = 1450,
  PASSWORDSPRIVATE_GETCOMPROMISEDCREDENTIALS = 1451,
  INPUTMETHODPRIVATE_HIDEINPUTVIEW = 1452,
  PASSWORDSPRIVATE_GETPLAINTEXTCOMPROMISEDPASSWORD = 1453,
  PASSWORDSPRIVATE_CHANGECOMPROMISEDCREDENTIAL = 1454,
  PASSWORDSPRIVATE_REMOVECOMPROMISEDCREDENTIAL = 1455,
  TERMINALPRIVATE_GETA11YSTATUS = 1456,
  AUTOTESTPRIVATE_GETSHELFUIINFOFORSTATE = 1457,
  PASSWORDSPRIVATE_STARTPASSWORDCHECK = 1458,
  PASSWORDSPRIVATE_STOPPASSWORDCHECK = 1459,
  PASSWORDSPRIVATE_GETPASSWORDCHECKSTATUS = 1460,
  TERMINALPRIVATE_OPENVMSHELLPROCESS = 1461,
  PASSWORDSPRIVATE_OPTINFORACCOUNTSTORAGE = 1462,
  CRYPTOTOKENPRIVATE_RECORDREGISTERREQUEST = 1463,
  CRYPTOTOKENPRIVATE_RECORDSIGNREQUEST = 1464,
  // Last entry: Add new entries above, then run:
  // python tools/metrics/histograms/update_extension_histograms.py
  ENUM_BOUNDARY
};

}  // namespace functions
}  // namespace extensions

#endif  // EXTENSIONS_BROWSER_EXTENSION_FUNCTION_HISTOGRAM_VALUE_H_
