// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_SECURITY_INTERSTITIALS_CORE_UNSAFE_RESOURCE_H_
#define COMPONENTS_SECURITY_INTERSTITIALS_CORE_UNSAFE_RESOURCE_H_

#include <vector>

#include "base/callback.h"
#include "base/memory/ref_counted.h"
#include "base/single_thread_task_runner.h"
#include "components/safe_browsing/core/db/hit_report.h"
#include "url/gurl.h"

namespace content {
class WebContents;
}  // namespace content

namespace security_interstitials {

// Structure that passes parameters between the IO and UI thread when
// interacting with the safe browsing blocking page.
struct UnsafeResource {
  // Passed booleans indicating whether or not it is OK to proceed with
  // loading an URL and whether or not an interstitial was shown as a result of
  // the URL load, |showed_interstitial| should only be set to true if the
  // interstitial was shown as a direct result of the navigation to the URL.
  // (e.g. it should be set to true if the interstitial will be shown from a
  // navigation throttle triggered by this navigation, but to false if it will
  // be shown using LoadPostCommitErrorPage).
  using UrlCheckCallback =
      base::RepeatingCallback<void(bool /*proceed*/,
                                   bool /*showed_interstitial*/)>;

  UnsafeResource();
  UnsafeResource(const UnsafeResource& other);
  ~UnsafeResource();

  // Returns true if this UnsafeResource is a main frame load that was blocked
  // while the navigation is still pending. Note that a main frame hit may not
  // be blocking, eg. client side detection happens after the load is
  // committed.
  bool IsMainPageLoadBlocked() const;

  GURL url;
  GURL original_url;
  GURL navigation_url;
  GURL referrer_url;
  std::vector<GURL> redirect_urls;
  bool is_subresource;
  bool is_subframe;
  safe_browsing::SBThreatType threat_type;
  safe_browsing::ThreatMetadata threat_metadata;
  UrlCheckCallback callback;  // This is called back on |callback_thread|.
  scoped_refptr<base::SingleThreadTaskRunner> callback_thread;
  base::RepeatingCallback<content::WebContents*(void)> web_contents_getter;
  safe_browsing::ThreatSource threat_source;
  // |token| field is only set if |threat_type| is
  // SB_THREAT_TYPE_*_PASSWORD_REUSE.
  std::string token;

  // If true, this UnsafeResource is created because of the Delayed Warnings
  // experiment.
  bool is_delayed_warning;
};

}  // namespace security_interstitials

#endif  // COMPONENTS_SECURITY_INTERSTITIALS_CORE_UNSAFE_RESOURCE_H_
