/*
** Copyright (C) 2015-2017,2019-2020 Red Hat.
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public License as published by the
** Free Software Foundation; either version 2, or (at your option) any
** later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU General Public License for more details.
*/

#include <pcp/pmapi.h>
#include <ctype.h>

#include "atop.h"
#include "photoproc.h"
#include "procmetrics.h"

/*
** sampled proc values into task structure, for one process/thread
*/
static void
update_task(struct tstat *task, int pid, char *name, pmResult *rp, pmDesc *dp, int offset)
{
	char *nametail = name;
	memset(task, 0, sizeof(struct tstat));

	strsep(&nametail, " ");	/* remove process identifier prefix; might fail */
	strncpy(task->gen.cmdline, nametail ? nametail : name, CMDLEN);
	task->gen.cmdline[CMDLEN] = '\0';
	task->gen.isproc = 1;		/* thread/process marker */

	/* accumulate Pss from smaps (optional, relatively expensive) */
	if (!calcpss)
	    task->mem.pmem = (unsigned long long)-1;
	else
	    task->mem.pmem = extract_ucount_t_inst(rp, dp, TASK_MEM_PMEM, pid, offset);

	/* /proc/pid/cgroup */
	extract_string_inst(rp, dp, TASK_GEN_CONTAINER, &task->gen.container[0],
				sizeof(task->gen.container), pid, offset);
        if (task->gen.container[0] != '\0')
		supportflags |= DOCKSTAT;

	/* /proc/pid/stat */
	extract_string_inst(rp, dp, TASK_GEN_NAME, &task->gen.name[0],
				sizeof(task->gen.name), pid, offset);
	extract_string_inst(rp, dp, TASK_GEN_STATE, &task->gen.state,
				sizeof(task->gen.state), pid, offset);

	task->gen.pid = extract_integer_inst(rp, dp, TASK_GEN_PID, pid, offset);
	task->gen.ppid = extract_integer_inst(rp, dp, TASK_GEN_PPID, pid, offset);
	if (task->gen.ppid <= 0 && pid != 1)
		task->gen.ppid = 1;
	task->mem.minflt = extract_count_t_inst(rp, dp, TASK_MEM_MINFLT, pid, offset);
	task->mem.majflt = extract_count_t_inst(rp, dp, TASK_MEM_MAJFLT, pid, offset);
	task->cpu.utime = extract_count_t_inst(rp, dp, TASK_CPU_UTIME, pid, offset);
	task->cpu.stime = extract_count_t_inst(rp, dp, TASK_CPU_STIME, pid, offset);
	task->cpu.prio = extract_integer_inst(rp, dp, TASK_CPU_PRIO, pid, offset);
	task->cpu.nice = extract_integer_inst(rp, dp, TASK_CPU_NICE, pid, offset);
	task->gen.btime = extract_integer_inst(rp, dp, TASK_GEN_BTIME, pid, offset);
	task->mem.vmem = extract_count_t_inst(rp, dp, TASK_MEM_VMEM, pid, offset);
	task->mem.rmem = extract_count_t_inst(rp, dp, TASK_MEM_RMEM, pid, offset);
	task->cpu.curcpu = extract_integer_inst(rp, dp, TASK_CPU_CURCPU, pid, offset);
	task->cpu.rtprio = extract_integer_inst(rp, dp, TASK_CPU_RTPRIO, pid, offset);
	task->cpu.policy = extract_integer_inst(rp, dp, TASK_CPU_POLICY, pid, offset);

	/* /proc/pid/status */
	task->gen.nthr = extract_integer_inst(rp, dp, TASK_GEN_NTHR, pid, offset);
	task->gen.tgid = extract_integer_inst(rp, dp, TASK_GEN_TGID, pid, offset);
	if (task->gen.tgid <= 0)
		task->gen.tgid = pid;
	task->gen.ctid = extract_integer_inst(rp, dp, TASK_GEN_ENVID, pid, offset);
	task->gen.vpid = extract_integer_inst(rp, dp, TASK_GEN_VPID, pid, offset);
	task->gen.ruid = extract_integer_inst(rp, dp, TASK_GEN_RUID, pid, offset);
	task->gen.euid = extract_integer_inst(rp, dp, TASK_GEN_EUID, pid, offset);
	task->gen.suid = extract_integer_inst(rp, dp, TASK_GEN_SUID, pid, offset);
	task->gen.fsuid = extract_integer_inst(rp, dp, TASK_GEN_FSUID, pid, offset);
	task->gen.rgid = extract_integer_inst(rp, dp, TASK_GEN_RGID, pid, offset);
	task->gen.egid = extract_integer_inst(rp, dp, TASK_GEN_EGID, pid, offset);
	task->gen.sgid = extract_integer_inst(rp, dp, TASK_GEN_SGID, pid, offset);
	task->gen.fsgid = extract_integer_inst(rp, dp, TASK_GEN_FSGID, pid, offset);
	task->mem.vdata = extract_count_t_inst(rp, dp, TASK_MEM_VDATA, pid, offset);
	task->mem.vstack = extract_count_t_inst(rp, dp, TASK_MEM_VSTACK, pid, offset);
	task->mem.vexec = extract_count_t_inst(rp, dp, TASK_MEM_VEXEC, pid, offset);
	task->mem.vlibs = extract_count_t_inst(rp, dp, TASK_MEM_VLIBS, pid, offset);
	task->mem.vswap = extract_count_t_inst(rp, dp, TASK_MEM_VSWAP, pid, offset);

	/* /proc/pid/io */
	task->dsk.rsz = extract_count_t_inst(rp, dp, TASK_DSK_RSZ, pid, offset);
	task->dsk.wsz = extract_count_t_inst(rp, dp, TASK_DSK_WSZ, pid, offset);
	task->dsk.cwsz = extract_count_t_inst(rp, dp, TASK_DSK_CWSZ, pid, offset);

	/*
 	** normalization
	*/
	task->cpu.prio   += 100; 	/* was subtracted by kernel */

	switch (task->gen.state)
	{
  	   case 'R':
		task->gen.nthrrun  = 1;
		break;
  	   case 'S':
		task->gen.nthrslpi = 1;
		break;
  	   case 'D':
		task->gen.nthrslpu = 1;
		break;
	}

	if (task->gen.tgid > 0 && task->gen.tgid != pid)
		task->gen.isproc = 0;

	if (task->dsk.rsz >= 0)
		supportflags |= IOSTAT;

	task->dsk.rio = task->dsk.rsz /= 512; /* sectors */
	task->dsk.wio = task->dsk.wsz /= 512; /* sectors */
	task->dsk.cwsz /= 512;		    /* sectors */
}

unsigned long
photoproc(struct tstat **tasks, unsigned int *taskslen)
{
	static int	setup;
	static pmID	pssid;
	static pmID	pmids[TASK_NMETRICS];
	static pmDesc	descs[TASK_NMETRICS];
	pmResult	*result;
	char		**insts;
	int		*pids, offset;
	unsigned long	count, i;

	if (!setup)
	{
		if (!hotprocflag)
			for (i = 0; i < TASK_NMETRICS; i++)
				procmetrics[i] += 3;	/* skip "hot" */
		setup_metrics(procmetrics, pmids, descs, TASK_NMETRICS);
		pssid = pmids[TASK_MEM_PMEM];
		setup = 1;

		/* check if per-process network metrics are available */
		netproc_probe();
	}

	if (!calcpss)
		pmids[TASK_MEM_PMEM] = PM_ID_NULL;
	else
		pmids[TASK_MEM_PMEM] = pssid;

	fetch_metrics("task", TASK_NMETRICS, pmids, &result);

	/* extract external process names (insts) */
	count = get_instances("task", TASK_GEN_NAME, descs, &pids, &insts);
	if (count > *taskslen)
	{
		size_t	size = count * sizeof(struct tstat);

		*tasks = (struct tstat *)realloc(*tasks, size);
		ptrverify(*tasks, "photoproc [%ld]\n", (long)size);
		*taskslen = count;
	}

	supportflags &= ~DOCKSTAT;

	for (i=0; i < count; i++)
	{
		if (pmDebugOptions.appl0)
			fprintf(stderr, "%s: updating process %d: %s\n",
				pmGetProgname(), pids[i], insts[i]);
		offset = get_instance_index(result, TASK_GEN_PID, pids[i]);
		update_task(&(*tasks)[i], pids[i], insts[i], result, descs, offset);
	}

	if (supportflags & NETATOP)
		netproc_update_tasks(tasks, count);

	if (pmDebugOptions.appl0)
		fprintf(stderr, "%s: done %lu processes\n", pmGetProgname(), count);

	pmFreeResult(result);
	if (count > 0) {
	    free(insts);
	    free(pids);
	}

	return count;
}
