// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"
	"time"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	forgejo_sdk "code.forgejo.org/f3/gof3/v3/forges/forgejo/sdk"
)

type milestone struct {
	common

	forgejoMilestone *forgejo_sdk.Milestone
}

var _ f3_tree.ForgeDriverInterface = &milestone{}

func newMilestone() generic.NodeDriverInterface {
	return &milestone{}
}

func (o *milestone) SetNative(milestone any) {
	o.forgejoMilestone = milestone.(*forgejo_sdk.Milestone)
}

func (o *milestone) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoMilestone.ID)
}

func (o *milestone) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

func (o *milestone) ToFormat() f3.Interface {
	if o.forgejoMilestone == nil {
		return o.NewFormat()
	}

	createdAT := time.Time{}
	var updatedAT *time.Time
	if o.forgejoMilestone.Closed != nil {
		createdAT = *o.forgejoMilestone.Closed
		updatedAT = o.forgejoMilestone.Closed
	}

	if !o.forgejoMilestone.Created.IsZero() {
		createdAT = o.forgejoMilestone.Created
	}
	if o.forgejoMilestone.Updated != nil && !o.forgejoMilestone.Updated.IsZero() {
		updatedAT = o.forgejoMilestone.Updated
	}

	return &f3.Milestone{
		Common:      f3.NewCommon(o.GetNativeID()),
		Title:       o.forgejoMilestone.Title,
		Description: o.forgejoMilestone.Description,
		Deadline:    o.forgejoMilestone.Deadline,
		Created:     createdAT,
		Updated:     updatedAT,
		Closed:      o.forgejoMilestone.Closed,
		State:       string(o.forgejoMilestone.State),
	}
}

func (o *milestone) FromFormat(content f3.Interface) {
	milestone := content.(*f3.Milestone)
	o.forgejoMilestone = &forgejo_sdk.Milestone{
		ID:          util.ParseInt(milestone.GetID()),
		Title:       milestone.Title,
		Description: milestone.Description,
		State:       forgejo_sdk.StateType(milestone.State),
		Created:     milestone.Created,
		Updated:     milestone.Updated,
		Closed:      milestone.Closed,
		Deadline:    milestone.Deadline,
	}
}

func (o *milestone) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	milestone, resp, err := o.getClient().GetMilestone(owner, project, node.GetID().Int64())
	if resp.StatusCode == 404 {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("milestone %v %w", o, err))
	}
	o.forgejoMilestone = milestone
	return true
}

func (o *milestone) Put(ctx context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	milestone, _, err := o.getClient().CreateMilestone(owner, project, forgejo_sdk.CreateMilestoneOption{
		Title:       o.forgejoMilestone.Title,
		Description: o.forgejoMilestone.Description,
		State:       o.forgejoMilestone.State,
		Deadline:    o.forgejoMilestone.Deadline,
	})
	if err != nil {
		panic(err)
	}
	o.forgejoMilestone = milestone
	return id.NewNodeID(o.GetNativeID())
}

func (o *milestone) Patch(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, _, err := o.getClient().EditMilestone(owner, project, node.GetID().Int64(), forgejo_sdk.EditMilestoneOption{
		Title:       o.forgejoMilestone.Title,
		Description: &o.forgejoMilestone.Description,
		State:       &o.forgejoMilestone.State,
		Deadline:    o.forgejoMilestone.Deadline,
	})
	if err != nil {
		panic(err)
	}
}

func (o *milestone) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())

	_, err := o.getClient().DeleteMilestone(owner, project, node.GetID().Int64())
	if err != nil {
		panic(err)
	}
}
