/** @file iobject.h  Interface for a Doomsday Script object.
 *
 * @authors Copyright (c) 2015-2017 Jaakko Keränen <jaakko.keranen@iki.fi>
 *
 * @par License
 * LGPL: http://www.gnu.org/licenses/lgpl.html
 *
 * <small>This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at your
 * option) any later version. This program is distributed in the hope that it
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details. You should have received a copy of
 * the GNU Lesser General Public License along with this program; if not, see:
 * http://www.gnu.org/licenses</small>
 */

#ifndef LIBDENG2_IOBJECT_H
#define LIBDENG2_IOBJECT_H

#include "../Record"

namespace de {

class Variable;

/**
 * Interface for a Doomsday Script object.
 *
 * Classes that implement this interface can be manipulated in scripts as
 * objects, because they have a Record that corresponds the native instance.
 *
 * Objects that implement IObject can be passed as arguments in
 * Process::scriptCall().
 *
 * @ingroup script
 */
class DENG2_PUBLIC IObject
{
public:
    virtual ~IObject() {}

    /**
     * Returns the Record that contains the instance namespace of the object.
     */
    virtual Record &objectNamespace() = 0;

    /// @copydoc objectNamespace()
    virtual Record const &objectNamespace() const = 0;

    /**
     * Looks up a variable in the object namespace. Variables in subrecords can
     * be accessed using the member notation: `subrecordName.variableName`
     *
     * If the variable does not exist, a Record::NotFoundError is thrown.
     *
     * @param name  Variable name.
     *
     * @return  Variable.
     */
    inline Variable &operator [] (String const &name) {
        return objectNamespace()[name];
    }

    /**
     * Looks up a variable in the object namespace. Variables in subrecords can
     * be accessed using the member notation: `subrecordName.variableName`
     *
     * If the variable does not exist, a Record::NotFoundError is thrown.
     *
     * @param name  Variable name.
     *
     * @return  Variable.
     */
    inline Variable const &operator [] (String const &name) const {
        return objectNamespace()[name];
    }
};

} // namespace de

#endif // LIBDENG2_IOBJECT_H

