/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include "cdw_widgets.h"
#include "cdw_debug.h"


/* Indexed with values of enum from header file. */
static const char *widget_type_labels[] = {
	"NONE",                      /* Right now the IDs in header file start with 1. */
	"STATIC_LABEL",
	"CHECKBOX",
	"DROPDOWN",
	"BUTTON",
	"TEXT",
	"SAFE_INPUT_LINE",
	"DYNAMIC_LABEL",
	"FILE_SELECTOR"
};





const char *cdw_widgets_widget_type_label(cdw_id_t widget_type)
{
	return widget_type_labels[widget_type];
}





bool cdw_widget_is_valid_widget_id(cdw_id_t widget_type)
{
	/* This assert is used to make sure that declarations in
	   header file are consistent with contents of *.c file. */
	cdw_assert (CDW_WIDGET_ID_INIT                == 0
		    && CDW_WIDGET_ID_STATIC_LABEL     == 1
		    && CDW_WIDGET_ID_CHECKBOX         == 2
		    && CDW_WIDGET_ID_DROPDOWN         == 3
		    && CDW_WIDGET_ID_BUTTON           == 4
		    && CDW_WIDGET_ID_TEXT             == 5
		    && CDW_WIDGET_ID_SAFE_INPUT_LINE  == 6
		    && CDW_WIDGET_ID_DYNAMIC_LABEL    == 7
		    && CDW_WIDGET_ID_FILE_SELECTOR    == 8,

		    "ERROR: changed order of widget IDs, check code that uses this function as the code may depend on order of the IDs\n");


	if (widget_type == CDW_WIDGET_ID_STATIC_LABEL
	    || widget_type == CDW_WIDGET_ID_CHECKBOX
	    || widget_type == CDW_WIDGET_ID_DROPDOWN
	    || widget_type == CDW_WIDGET_ID_BUTTON
	    || widget_type == CDW_WIDGET_ID_TEXT
	    || widget_type == CDW_WIDGET_ID_SAFE_INPUT_LINE
	    || widget_type == CDW_WIDGET_ID_DYNAMIC_LABEL
	    || widget_type == CDW_WIDGET_ID_FILE_SELECTOR) {

		return true;

	} else {
		return false;
	}
}





/**
   \brief Initialize cdw_widget_t variable

   \date Function's top-level comment reviewed on 2016-02-18
   \date Function's body reviewed on 2016-02-18

   \param widget - variable to initialize
*/
void cdw_widget_init(cdw_widget_t *widget)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");

	widget->type_id = CDW_WIDGET_ID_INIT;

	for (int i = 0; i < CDW_WIDGET_N_RETURN_KEYS_MAX; i++) {
		widget->return_keys[i] = 0;
	}

	widget->n_return_keys = 0;

	widget->add_return_key = NULL;

	widget->self = NULL;

	return;
}





/**
   \brief Add key that, if captured by widget's driver, will cause return of control by driver

   \date Function's top-level comment reviewed on 2016-02-18
   \date Function's body reviewed on 2016-02-18

   Function registers a key \p key, which, if captured by driver, will cause
   the driver to return control to it's (driver's) caller.

   You may want to use this function e.g. when you want to perform some
   action whenever ENTER key is hit. In that case you have to configure
   a widget before using it by calling this function with CDW_KEY_ENTER as
   second argument. Similarly for other keys: Space, Delete, any digit or
   letter. Digits should be passed as characters.

   \param widget - widget that you want to configure
   \param key - key that you want to add to given \p widget
*/
void cdw_widget_add_return_key(cdw_widget_t *widget, int key)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (widget->n_return_keys < CDW_WIDGET_N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the widget, can't add another one\n",
		    widget->n_return_keys, CDW_WIDGET_N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key == 0, but 0 is an initializer value\n");

	if (cdw_widget_is_return_key(widget, key)) {
		cdw_sdm ("WARNING: attempting to add key %d / \"%s\", but it is already on the list of return keys\n",
			 key, cdw_ncurses_key_label(key));
	} else {
		widget->return_keys[widget->n_return_keys++] = key;
	}

	return;
}





/**
   \brief Add key(s) that, if captured by widget's driver, will cause return of control by driver

   \date Function's top-level comment reviewed on 2016-02-18
   \date Function's body reviewed on 2016-02-18

   Function registers keys, which, if captured by widget's driver, will cause
   the driver to return control to it's (driver's) caller.

   You may want to use this function e.g. when you want to perform some
   action whenever ENTER key is hit. In that case you have to configure
   a widget before using it by calling this function with CDW_KEY_ENTER as
   second argument. Similarly for other keys: Space, Delete, any digit or
   letter. Digits should be passed as characters.

   \param widget - widget that you want to configure
   \param ... - list of keys to add; use '0' as a last key (guard)
*/
void cdw_widget_add_return_keys(cdw_widget_t *widget, ...)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (widget->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    widget->n_return_keys, N_RETURN_KEYS_MAX);

	va_list ap;
	va_start(ap, widget);
	int key = 'a';
	while ((key = va_arg(ap, int)) != 0) {
		cdw_widget_add_return_key(widget, key);
	}
	va_end(ap);

	return;
}





/**
   \brief Add key(s) that, if captured by widget's driver, will cause return of control by driver

   \date Function's top-level comment reviewed on 2016-03-20
   \date Function's body reviewed on 2016-03-20

   Function registers keys, which, if captured by widget's driver, will cause
   the driver to return control to it's (driver's) caller.

   You may want to use this function e.g. when you want to perform some
   action whenever ENTER key is hit. In that case you have to configure
   a widget before using it by calling this function with CDW_KEY_ENTER as
   second argument. Similarly for other keys: Space, Delete, any digit or
   letter. Digits should be passed as characters.
   
   This function uses widget->add_return_key() function registered for
   a \p widget.

   \param widget - widget that you want to configure
   \param ... - list of keys to add; use '0' as a last key (guard)
*/
void cdw_widget_add_return_keys_pointer(cdw_widget_t *widget, ...)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (widget->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    widget->n_return_keys, N_RETURN_KEYS_MAX);
	cdw_assert (widget->add_return_key, "ERROR: pointer to function is NULL\n");

	va_list ap;
	va_start(ap, widget);
	int key = 'a';
	while ((key = va_arg(ap, int)) != 0) {
		widget->add_return_key(widget, key);
	}
	va_end(ap);

	return;
}





/**
  \brief Check if given key is one of configured "return" keys

  \date Function's top-level comment reviewed on 2016-02-18
  \date Function's body reviewed on 2016-02-18

  Function checks if given \p key was earlier added to given \p widget
  as "return on the key" key.

  \param widget - widget that you want to query
  \param key - key you want to check if it is added to widget

  \return true if given key was added to given \p widget
  \return false if given key was not added to given \p widget
*/
bool cdw_widget_is_return_key(cdw_widget_t const * widget, int key)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (key != 0, "ERROR: asking for 0, which is an initialization value\n");

	for (int i = 0; i < widget->n_return_keys; i++) {
		if (widget->return_keys[i] == key) {
			return true;
		}
	}

	return false;
}
