"""
Determine the wheel name via setuptools

Usage:
$ python3 wheelname.py <package-name> <version-string>
"""

from setuptools import Extension, Distribution
import platform

HEADER = "wheelname.py"

def wheel_tags(**kwargs):
    """ set up a dummy distribution from arguments
    and return a resulting wheel name
    """

    dst = Distribution(attrs=kwargs)
    # finalize bdist_wheel command
    bdist_wheel_cmd = dst.get_command_obj('bdist_wheel')
    bdist_wheel_cmd.ensure_finalized()

    distname = bdist_wheel_cmd.wheel_dist_name  # eg. 'testdist-1.2.3'
    tags = bdist_wheel_cmd.get_tag() # eg. (cp311, cp311, linux_x86_64)

    return (distname, *tags)


def wheel_name(*tags):
    # eg. 'testdist-1.2.3-cp311-cp311-linux_x86_64'
    return '-'.join(*tags)


def get_wheel_names(pkg_name:str, version_str:str):
    # get the name of the pure-Python wheel
    pure_tags = wheel_tags(name=pkg_name, version=version_str)
    pure_whl = wheel_name(pure_tags)

    # get tags for the platform-dependent wheel
    platform_tags = wheel_tags(name=pkg_name, version=version_str,
                               ext_modules=[Extension("dummylib", ["dummy.c"])])
    # fix platform tag to be compatible with conda under MacOS-x64
    platform_whl = wheel_name(platform_tags)

    # NOTE: CMake list separator is semicolon
    return ';'.join((pure_whl, platform_whl))

#----------------------------------------

if __name__ == "__main__":
    import sys

    args = sys.argv
    if len(args) <= 2:
        print("Usage: python3 wheelname.py <package-name> <version-string>")
        raise ValueError(HEADER + ": package name and version string must be non-empty.")

    """ NOTE:
    Python pip currently enforces PEP8 rule that Python packages
    should also have short, all-lowercase names.
    <https://peps.python.org/pep-0008/#package-and-module-names>
    """

    pkg_name = args[1].strip().lower()
    version_str = args[2].strip()
    print(get_wheel_names(pkg_name, version_str))
