﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ram/RAMRequest.h>
#include <aws/ram/RAM_EXPORTS.h>
#include <aws/ram/model/Tag.h>

#include <utility>

namespace Aws {
namespace RAM {
namespace Model {

/**
 */
class CreateResourceShareRequest : public RAMRequest {
 public:
  AWS_RAM_API CreateResourceShareRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateResourceShare"; }

  AWS_RAM_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Specifies the name of the resource share.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateResourceShareRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a list of one or more ARNs of the resources to associate with the
   * resource share.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceArns() const { return m_resourceArns; }
  inline bool ResourceArnsHasBeenSet() const { return m_resourceArnsHasBeenSet; }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  void SetResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns = std::forward<ResourceArnsT>(value);
  }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  CreateResourceShareRequest& WithResourceArns(ResourceArnsT&& value) {
    SetResourceArns(std::forward<ResourceArnsT>(value));
    return *this;
  }
  template <typename ResourceArnsT = Aws::String>
  CreateResourceShareRequest& AddResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns.emplace_back(std::forward<ResourceArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a list of one or more principals to associate with the resource
   * share.</p> <p>You can include the following values:</p> <ul> <li> <p>An Amazon
   * Web Services account ID, for example: <code>123456789012</code> </p> </li> <li>
   * <p>An <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a> of an organization in Organizations, for example:
   * <code>organizations::123456789012:organization/o-exampleorgid</code> </p> </li>
   * <li> <p>An ARN of an organizational unit (OU) in Organizations, for example:
   * <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
   * </p> </li> <li> <p>An ARN of an IAM role, for example:
   * <code>iam::123456789012:role/rolename</code> </p> </li> <li> <p>An ARN of an IAM
   * user, for example: <code>iam::123456789012user/username</code> </p> </li> <li>
   * <p>A service principal name, for example: <code>service-id.amazonaws.com</code>
   * </p> </li> </ul>  <p>Not all resource types can be shared with IAM roles
   * and users. For more information, see <a
   * href="https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types">Sharing
   * with IAM roles and users</a> in the <i>Resource Access Manager User
   * Guide</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPrincipals() const { return m_principals; }
  inline bool PrincipalsHasBeenSet() const { return m_principalsHasBeenSet; }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  void SetPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals = std::forward<PrincipalsT>(value);
  }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  CreateResourceShareRequest& WithPrincipals(PrincipalsT&& value) {
    SetPrincipals(std::forward<PrincipalsT>(value));
    return *this;
  }
  template <typename PrincipalsT = Aws::String>
  CreateResourceShareRequest& AddPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals.emplace_back(std::forward<PrincipalsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies one or more tags to attach to the resource share itself. It doesn't
   * attach the tags to the resources associated with the resource share.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateResourceShareRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateResourceShareRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether principals outside your organization in Organizations can
   * be associated with a resource share. A value of <code>true</code> lets you share
   * with individual Amazon Web Services accounts that are <i>not</i> in your
   * organization. A value of <code>false</code> only has meaning if your account is
   * a member of an Amazon Web Services Organization. The default value is
   * <code>true</code>.</p>
   */
  inline bool GetAllowExternalPrincipals() const { return m_allowExternalPrincipals; }
  inline bool AllowExternalPrincipalsHasBeenSet() const { return m_allowExternalPrincipalsHasBeenSet; }
  inline void SetAllowExternalPrincipals(bool value) {
    m_allowExternalPrincipalsHasBeenSet = true;
    m_allowExternalPrincipals = value;
  }
  inline CreateResourceShareRequest& WithAllowExternalPrincipals(bool value) {
    SetAllowExternalPrincipals(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. This lets you safely retry the request without
   * accidentally performing the same operation a second time. Passing the same value
   * to a later call to an operation requires that you also pass the same value for
   * all other parameters. We recommend that you use a <a
   * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID type of
   * value.</a>.</p> <p>If you don't provide this value, then Amazon Web Services
   * generates a random one for you.</p> <p>If you retry the operation with the same
   * <code>ClientToken</code>, but with different parameters, the retry fails with an
   * <code>IdempotentParameterMismatch</code> error.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateResourceShareRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a> of the RAM permission to associate with the resource
   * share. If you do not specify an ARN for the permission, RAM automatically
   * attaches the default version of the permission for each resource type. You can
   * associate only one permission with each resource type included in the resource
   * share.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPermissionArns() const { return m_permissionArns; }
  inline bool PermissionArnsHasBeenSet() const { return m_permissionArnsHasBeenSet; }
  template <typename PermissionArnsT = Aws::Vector<Aws::String>>
  void SetPermissionArns(PermissionArnsT&& value) {
    m_permissionArnsHasBeenSet = true;
    m_permissionArns = std::forward<PermissionArnsT>(value);
  }
  template <typename PermissionArnsT = Aws::Vector<Aws::String>>
  CreateResourceShareRequest& WithPermissionArns(PermissionArnsT&& value) {
    SetPermissionArns(std::forward<PermissionArnsT>(value));
    return *this;
  }
  template <typename PermissionArnsT = Aws::String>
  CreateResourceShareRequest& AddPermissionArns(PermissionArnsT&& value) {
    m_permissionArnsHasBeenSet = true;
    m_permissionArns.emplace_back(std::forward<PermissionArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies source constraints (accounts, ARNs, organization IDs, or
   * organization paths) that limit when service principals can access resources in
   * this resource share. When a service principal attempts to access a shared
   * resource, validation is performed to ensure the request originates from one of
   * the specified sources. This helps prevent confused deputy attacks by applying
   * constraints on where service principals can access resources from.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSources() const { return m_sources; }
  inline bool SourcesHasBeenSet() const { return m_sourcesHasBeenSet; }
  template <typename SourcesT = Aws::Vector<Aws::String>>
  void SetSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources = std::forward<SourcesT>(value);
  }
  template <typename SourcesT = Aws::Vector<Aws::String>>
  CreateResourceShareRequest& WithSources(SourcesT&& value) {
    SetSources(std::forward<SourcesT>(value));
    return *this;
  }
  template <typename SourcesT = Aws::String>
  CreateResourceShareRequest& AddSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources.emplace_back(std::forward<SourcesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::Vector<Aws::String> m_resourceArns;

  Aws::Vector<Aws::String> m_principals;

  Aws::Vector<Tag> m_tags;

  bool m_allowExternalPrincipals{false};

  Aws::String m_clientToken;

  Aws::Vector<Aws::String> m_permissionArns;

  Aws::Vector<Aws::String> m_sources;
  bool m_nameHasBeenSet = false;
  bool m_resourceArnsHasBeenSet = false;
  bool m_principalsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_allowExternalPrincipalsHasBeenSet = false;
  bool m_clientTokenHasBeenSet = false;
  bool m_permissionArnsHasBeenSet = false;
  bool m_sourcesHasBeenSet = false;
};

}  // namespace Model
}  // namespace RAM
}  // namespace Aws
